# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Data for entertainment streaming services
streaming_services = {
    "FlixHaven": np.random.normal(loc=(100, 80), scale=10, size=(50, 2)),
    "CinemaNow": np.random.normal(loc=(120, 85), scale=10, size=(45, 2)),
    "StreamIt": np.random.normal(loc=(90, 75), scale=15, size=(60, 2)),
    "MovieFlow": np.random.normal(loc=(110, 90), scale=12, size=(55, 2)),
    "PlayBox": np.random.normal(loc=(130, 95), scale=10, size=(35, 2)),
    "EntertainMax": np.random.normal(loc=(105, 88), scale=14, size=(50, 2)),
}

title = "Streaming Service Performance Across Viewing Time and Satisfaction"
xlabel = "Average Streaming Time (Minutes)"
ylabel = "Viewer Satisfaction Score"
# Inset plot configuration
insetaxes = [0.2, 0.6, 0.3, 0.3]
insetxlim = [85, 115]  # Focus on FlixHaven and StreamIt
insetylim = [70, 90]
insetxticks = [85, 95, 105, 115]
insetyticks = [70, 75, 80, 85, 90]

# Arrow and annotation configuration
arrowstart = (100, 90)  # Arrow start for FlixHaven
arrowend = (0.34, 0.72)  # Relative inset arrow end
annotaterecx = [85, 115]  # Annotation range for X
annotaterecy = [70, 90]  # Annotation range for Y

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors for streaming services
colors = {
    "FlixHaven": "#FF5733",
    "CinemaNow": "#33FF57",
    "StreamIt": "#3357FF",
    "MovieFlow": "#FF33A1",
    "PlayBox": "#9333FF",
    "EntertainMax": "#33FFF6",
}

# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 8))
for service, data in streaming_services.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[service],
        label=service,
        alpha=0.6,
        edgecolors="w",
        s=100,
        marker="o",  # Circle marker for better visibility
    )

# Enclosing rectangle for annotation
rect = plt.Rectangle(
    (annotaterecx[0], annotaterecy[0]),
    annotaterecx[1] - annotaterecx[0],
    annotaterecy[1] - annotaterecy[0],
    linewidth=1,
    edgecolor="black",
    facecolor="none",
    linestyle="--",  # Dashed line for rectangle
)
ax.add_patch(rect)

# Create the inset with zoomed-in view
ax_inset = fig.add_axes(insetaxes)
for service, data in streaming_services.items():
    ax_inset.scatter(
        data[:, 0], data[:, 1], c=colors[service], alpha=0.6, edgecolors="w", s=100
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=1),
)

# Adding titles and labels
ax.set_title(title, fontsize=16)
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)
ax.legend(title="Streaming Services", title_fontsize="13")

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_77.pdf", bbox_inches="tight")
