# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
subjects = {
    "Math": np.random.normal(loc=75, scale=10, size=(50, 2)),
    "Science": np.random.normal(loc=80, scale=8, size=(50, 2)),
    "English": np.random.normal(loc=70, scale=9, size=(50, 2)),
    "History": np.random.normal(loc=65, scale=12, size=(50, 2)),
    "Art": np.random.normal(loc=85, scale=7, size=(50, 2)),
}

insetaxes = [0.15, 0.15, 0.2, 0.2]
insetxlim = [60, 90]
insetylim = [55, 95]
insetxticks = [60, 75, 90]
insetyticks = [55, 75, 95]
arrowstart = (62, 65)
arrowend = (0.2, 0.2)
annotaterecx = [60, 90]
annotaterecy = [55, 95]
title = "Subject Performance Analysis"
xlabel = "Average Score"
ylabel = "Standard Deviation"
legendlabels = ["Math", "Science", "English", "History", "Art"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 10))

# Define colors and markers for each subject
colors = {
    "Math": "#1f77b4",  # Blue
    "Science": "#ff7f0e",  # Orange
    "English": "#2ca02c",  # Green
    "History": "#d62728",  # Red
    "Art": "#9467bd",  # Purple
}

markers = {
    "Math": "o",
    "Science": "^",
    "English": "s",
    "History": "p",
    "Art": "D",
}

for subject, data in subjects.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[subject],
        marker=markers[subject],
        label=subject,
        alpha=0.7,
        edgecolors="w",
    )

# Draw annotation rectangle around specific region
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with a zoomed-in view of a specific region
ax_inset = fig.add_axes(insetaxes)
for subject, data in subjects.items():
    ax_inset.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[subject],
        marker=markers[subject],
        alpha=0.7,
        edgecolors="w",
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add annotation to indicate the inset region
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# Add titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend(legendlabels)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("PIP_78.pdf", bbox_inches="tight")
