# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Simulated sports data (average speed vs endurance)
sports_activities = {
    "running": np.random.normal(loc=(10, 70), scale=5, size=(50, 2)),  # Running
    "cycling": np.random.normal(loc=(25, 65), scale=5, size=(30, 2)),  # Cycling
    "swimming": np.random.normal(loc=(5, 60), scale=5, size=(40, 2)),  # Swimming
    "hiking": np.random.normal(loc=(3, 80), scale=5, size=(60, 2)),  # Hiking
    "rowing": np.random.normal(loc=(15, 75), scale=5, size=(70, 2)),  # Rowing
}

insetaxes = [0.1, 0.15, 0.2, 0.2]
insetxlim = [0, 30]
insetylim = [50, 90]
insetxticks = [0, 15, 30]
insetyticks = [50, 70, 90]
arrowstart = (5, 60)  # Hiking
arrowend = (0.1, 0.1)
annotaterecx = [0, 30]
annotaterecy = [50, 90]
title = "Sports Activities Analysis"
xlabel = "Average Speed (km/h)"
ylabel = "Endurance (minutes)"
legendlabels = ["Running", "Cycling", "Swimming", "Hiking", "Rowing"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 10))

# Define colors and markers for each sports activity
colors = {
    "running": "#FF5733",  # Running
    "cycling": "#33FF57",  # Cycling
    "swimming": "#3357FF",  # Swimming
    "hiking": "#FF33B5",  # Hiking
    "rowing": "#FFC300",  # Rowing
}

markers = {
    "running": "o",  # Running
    "cycling": "s",  # Cycling
    "swimming": "^",  # Swimming
    "hiking": "d",  # Hiking
    "rowing": "p",  # Rowing
}

for activity, data in sports_activities.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[activity],
        marker=markers[activity],
        label=activity,
        alpha=0.7,
    )

# Draw annotation rectangle around specific region
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with a zoomed-in view of a specific region
ax_inset = fig.add_axes(insetaxes)
for activity, data in sports_activities.items():
    ax_inset.scatter(
        data[:, 0], data[:, 1], c=colors[activity], marker=markers[activity], alpha=0.7
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add annotation to indicate the inset region
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# Add titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend(legendlabels)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("PIP_79.pdf", bbox_inches="tight")
