
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Simulated data for finance domain
time_step = np.linspace(0, 24, 48)
stock_price = 100 + np.cumsum(np.random.normal(0.5, 2, 48))  # Stock prices with upward drift
trading_volume = 1000 + 500 * np.sin(np.pi * time_step / 12) + np.random.normal(0, 100, 48)

# Extracted variables for labels and settings
stock_price_label = "Stock Price (USD)"
trading_volume_label = "Trading Volume (Shares)"
stock_price_ylabel = "Stock Price (USD)"
trading_volume_xlabel = "Time (hours)"
trading_volume_ylabel = "Trading Volume (Shares)"
legend_labels = ["Stock Price", "Trading Volume"]

# Plot configuration parameters
stock_price_ylim = [min(stock_price) - 10, max(stock_price) + 10]
stock_price_xlim = [0, 24]
trading_volume_ylim = [min(trading_volume) - 200, max(trading_volume) + 200]
trading_volume_xlim = [0, 24]
stock_price_yticks = np.arange(min(stock_price) // 10 * 10, max(stock_price) // 10 * 10 + 20, 10)
trading_volume_yticks = np.arange(min(trading_volume) // 500 * 500, max(trading_volume) // 500 * 500 + 1000, 500)
time_xticks = np.arange(0, 25, 4)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

# Create figure and axes
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 10))

# Plot Stock Price
ax1.plot(time_step, stock_price, "o-", color="#1f77b4", label=stock_price_label)
ax1.fill_between(time_step, stock_price, color="#aec7e8", alpha=0.3)
ax1.set_ylim(stock_price_ylim)
ax1.set_xlim(stock_price_xlim)
ax1.set_yticks(stock_price_yticks)
ax1.set_xticks(time_xticks)
ax1.set_ylabel(stock_price_ylabel)
ax1.tick_params(axis="both", which="both", length=0)
ax1.grid(True)

# Plot Trading Volume
ax2.plot(time_step, trading_volume, "s--", color="#ff7f0e", label=trading_volume_label)
ax2.fill_between(time_step, trading_volume, color="#ffbb78", alpha=0.3)
ax2.set_ylim(trading_volume_ylim)
ax2.set_xlim(trading_volume_xlim)
ax2.set_yticks(trading_volume_yticks)
ax2.set_xticks(time_xticks)
ax2.set_xlabel(trading_volume_xlabel)
ax2.set_ylabel(trading_volume_ylabel)
ax2.tick_params(axis="both", which="both", length=0)
ax2.grid(True)

# Create custom legend
blue_line = mlines.Line2D(
    [], [], color="#1f77b4", marker="o", markersize=6, label=legend_labels[0]
)
orange_line = mlines.Line2D(
    [], [], color="#ff7f0e", marker="s", markersize=6, label=legend_labels[1]
)
plt.legend(
    handles=[blue_line, orange_line],
    loc="lower center",
    bbox_to_anchor=(0.5, -0.2),
    ncol=2,
    frameon=False,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("area_80.pdf", bbox_inches="tight")
