
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.ticker as ticker

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# New Data Generation
np.random.seed(42)  # For reproducibility in this example
# Sample data
brightness_values = np.linspace(0.5, 1.5, 11)
scale_values = np.linspace(0.75, 1.25, 11)
rotation_values = np.linspace(-150, 150, 11)

# Simulated gain change rates for 'Method A' and 'Method B'
gain_change_brightness_method_a = np.random.uniform(-25, 5, 11)
gain_change_scale_method_a = np.random.uniform(-45, 15, 11)
gain_change_rotation_method_a = np.random.uniform(-15, 35, 11)

gain_change_brightness_method_b = np.random.uniform(-20, 10, 11)
gain_change_scale_method_b = np.random.uniform(-40, 20, 11)
gain_change_rotation_method_b = np.random.uniform(-10, 30, 11)

# New labels and text parameters
label_method_a = "Method A"
label_method_b = "Method B"
xlabel_brightness = "Brightness Levels"
ylabel_gain_change = "Rate of Gain Change [%]"
xlabel_scale = "Scale Factors"
xlabel_rotation = "Rotation Angles [°]"
suptitle_text = "Comparison of Gain Change Rates by Methods A and B"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
ylim_brightness = (-30, 10)
xlim_brightness = (0.45, 1.55)
yticks_brightness = [-30, -20, -10, 0, 10]
xticks_brightness = np.linspace(0.5, 1.5, 11)

ylim_scale = (-50, 20)
xlim_scale = (0.73, 1.27)
yticks_scale = [-50, -40, -30, -20, -10, 0, 10, 20]
xticks_scale = np.linspace(0.75, 1.25, 11)

ylim_rotation = (-20, 40)
xlim_rotation = (-165, 165)
yticks_rotation = [-20, -10, 0, 10, 20, 30, 40]
xticks_rotation = np.linspace(-150, 150, 11)

# Create figure and subplots
fig, axs = plt.subplots(3, 1, figsize=(6, 10))
fig.suptitle(suptitle_text, fontsize=16)

# Color scheme for the bars
color_method_a = "#1f77b4"
color_method_b = "#ff7f0e"

# Top chart - Brightness
axs[0].bar(
    brightness_values - 0.02,
    gain_change_brightness_method_a,
    width=0.04,
    zorder=10,
    color=color_method_a,
    label=label_method_a,
)
axs[0].bar(
    brightness_values + 0.02,
    gain_change_brightness_method_b,
    width=0.04,
    zorder=10,
    color=color_method_b,
    label=label_method_b,
)
axs[0].set_xlabel(xlabel_brightness)
axs[0].set_ylabel(ylabel_gain_change)
axs[0].set_ylim(ylim_brightness)
axs[0].set_xlim(xlim_brightness)
axs[0].set_yticks(yticks_brightness)
axs[0].xaxis.set_major_locator(ticker.FixedLocator(xticks_brightness))
axs[0].xaxis.set_major_formatter(ticker.FuncFormatter(lambda x, _: f"{x:.1f}"))
offsetticks2 = [0.45] + [i + 0.05 for i in xticks_brightness]
axs[0].xaxis.set_minor_locator(ticker.FixedLocator(offsetticks2))
axs[0].grid(True, which="minor", axis="x", color="gray", linestyle='--', lw=0.5)
axs[0].grid(True, which="major", axis="y", color="gray", linestyle='-', lw=0.7)
axs[0].tick_params(axis="x", which="major", length=0)

# Middle chart - Scale
axs[1].bar(
    scale_values - 0.01,
    gain_change_scale_method_a,
    width=0.02,
    zorder=10,
    color=color_method_a,
)
axs[1].bar(
    scale_values + 0.01,
    gain_change_scale_method_b,
    width=0.02,
    zorder=10,
    color=color_method_b,
)
axs[1].set_xlabel(xlabel_scale)
axs[1].set_ylabel(ylabel_gain_change)
axs[1].set_ylim(ylim_scale)
axs[1].set_xlim(xlim_scale)
axs[1].set_yticks(yticks_scale)
axs[1].xaxis.set_major_locator(ticker.FixedLocator(xticks_scale))
axs[1].xaxis.set_major_formatter(ticker.FuncFormatter(lambda x, _: f"{x:.2f}"))
offsetticks1 = [0.725] + [i + 0.025 for i in xticks_scale]
axs[1].xaxis.set_minor_locator(ticker.FixedLocator(offsetticks1))
axs[1].grid(True, which="minor", axis="x", color="gray", linestyle='--', lw=0.5)
axs[1].grid(True, which="major", axis="y", color="gray", linestyle='-', lw=0.7)
axs[1].tick_params(axis="x", which="major", length=0)

# Bottom chart - Rotation Angle
axs[2].bar(
    rotation_values - 5,
    gain_change_rotation_method_a,
    width=10,
    zorder=10,
    color=color_method_a,
)
axs[2].bar(
    rotation_values + 5,
    gain_change_rotation_method_b,
    width=10,
    zorder=10,
    color=color_method_b,
)
axs[2].set_xlabel(xlabel_rotation)
axs[2].set_ylabel(ylabel_gain_change)
axs[2].set_ylim(ylim_rotation)
axs[2].set_xlim(xlim_rotation)
axs[2].set_yticks(yticks_rotation)
axs[2].tick_params(axis="x", which="major", length=0)

axs[2].xaxis.set_major_locator(ticker.FixedLocator(xticks_rotation))
axs[2].xaxis.set_major_formatter(ticker.FuncFormatter(lambda x, _: f"{int(x)}"))
offsetticks = [-165, -135, -105, -75, -45, -15, 15, 45, 75, 105, 135, 165]
axs[2].xaxis.set_minor_locator(ticker.FixedLocator(offsetticks))
axs[2].grid(True, which="minor", axis="x", color="gray", linestyle='--', lw=0.5)
axs[2].grid(True, which="major", axis="y", color="gray", linestyle='-', lw=0.7)

# Add legend
fig.legend(loc="upper center", ncol=2, bbox_to_anchor=(0.5, 1.03), frameon=False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout(rect=[0, 0, 1, 0.97])
plt.savefig("bar_144.pdf", bbox_inches="tight")
