
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
crops = ["Wheat", "Corn", "Soybean"]
yield_increase = {
    "0%": [100, 100, 100],
    "25%": [120, 135, 110],
    "50%": [150, 160, 140],
    "75%": [180, 190, 170],
    "100%": [210, 220, 200],
}

# Plot labels and types
ylabel_text = "Yield Increase (%)"
xlabel_text = "Crops"
title_text = "Crop Yield Increase Under Different Irrigation Conditions"
legend_title = "Irrigation Increase"

# Plot limits and ticks
ylim_values = (0, 240)
yticks_values = [0, 50, 100, 150, 200, 240]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
bar_label_fontsize = 10
ylabel_fontsize = 12

# Colors for the bars
colors = ["#8FBC8F", "#DAA520", "#556B2F", "#8B4513", "#DEB887"]

# Setup the figure and axes
fig, ax = plt.subplots(figsize=(12, 4))

# Bar width
bar_width = 0.1

# Positions of the bars on the x-axis
r = np.arange(len(crops))

# Create bars for each irrigation level
for i, (irrigation, values) in enumerate(yield_increase.items()):
    ax.bar(
        r + i * bar_width,
        values,
        color=colors[i],
        width=bar_width,
        edgecolor="white",
        label=irrigation,
    )

# Add labels on top of the bars
for i, (irrigation, values) in enumerate(yield_increase.items()):
    for j, value in enumerate(values):
        ax.text(
            j + i * bar_width,
            value + 5,
            str(value),
            ha="center",
            va="bottom",
            fontsize=bar_label_fontsize,
        )

# General layout
ax.set_ylabel(ylabel_text, fontsize=ylabel_fontsize)
ax.set_xticks(r + bar_width * (len(yield_increase) - 1) / 2)
ax.set_xticklabels(crops)
ax.set_ylim(*ylim_values)
ax.set_yticks(yticks_values)
ax.legend(
    title=legend_title,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.2),
    frameon=False,
    ncol=5,
)
ax.set_facecolor("#eaeaf2")
ax.yaxis.grid(True, color="white")
ax.set_axisbelow(True)

# Remove spines
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(False)
ax.spines["left"].set_visible(False)

plt.tick_params(axis="both", which="both", length=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(12, 4)
plt.tick_params(axis="both", which="both", length=0)
plt.tight_layout()
plt.savefig("bar_191.pdf", bbox_inches="tight")
