
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Define the data and category names for a geographical survey
category_names = [
    "Least Prefer",
    "Less Prefer",
    "Neutral",
    "Prefer",
    "Strongly Prefer",
]

results = {
    "North America": [5, 10, 15, 35, 35],
    "South America": [10, 12, 20, 28, 30],
    "Europe": [8, 15, 25, 20, 32],
    "Asia": [12, 18, 20, 25, 25],
    "Africa": [10, 10, 20, 30, 30],
}

ylim = [-90, 90]
yticks = np.arange(-90, 91, 20)
xlabel = "Survey Responses"
title = "Geographical Survey: Preferred Travel Destinations"
axhline = 0

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
def create_vertical_bar_chart(results, category_names):
    fig, ax = plt.subplots(figsize=(10, 6))

    labels = list(results.keys())
    data = np.array(list(results.values()))
    data_cum = data.cumsum(axis=1)
    middle_index = data.shape[1] // 2
    offsets = data[:, range(middle_index)].sum(axis=1) + data[:, middle_index] / 2

    # Optimized Color Mapping
    category_colors = [
        "#FF9999", "#FFD700", "#98FB98", "#87CEFA", "#DB7093"
    ]

    # Plot Bars
    for i, (colname, color) in enumerate(zip(category_names, category_colors)):
        heights = data[:, i]
        bottoms = data_cum[:, i] - heights - offsets
        rects = ax.bar(
            labels,
            heights,
            bottom=bottoms,
            width=0.5,
            label=colname,
            color=color,
            edgecolor="black",
        )
        for j, (bottom, height) in enumerate(zip(bottoms, heights)):
            text_x = bottom + height / 2
            text_y = j
            ax.text(
                text_y,
                text_x,
                f"{abs(height):.1f}%",
                va="center",
                ha="center",
                color="black",
                fontsize=8,
            )
    
    # Add Zero Reference Line
    ax.axhline(axhline, linestyle="--", color="black", alpha=0.25)

    # X Axis
    ax.set_ylim(ylim)
    ax.set_yticks(yticks)
    ax.yaxis.set_major_formatter(lambda y, pos: str(abs(int(y))))
    ax.set_xlabel(xlabel)
    ax.set_title(title)

    # Remove spines
    ax.spines["top"].set_visible(False)
    ax.spines["right"].set_visible(False)

    # Legend
    ax.legend(
        ncol=len(category_names) // 2,
        bbox_to_anchor=(0.5, 1.1),
        loc="upper center",
        frameon=False,
    )

    return fig, ax

fig, ax = create_vertical_bar_chart(results, category_names)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("bar_315.pdf", bbox_inches="tight")
