
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data for psychological study
data1 = [
    np.random.normal(50, std, 50)
    for std in [5, 8, 6, 4, 7]
]
data2 = [
    np.random.normal(45, std, 50)
    for std in [6, 9, 5, 6, 8]
]
data3 = [
    np.random.normal(55, std, 50)
    for std in [7, 10, 8, 5, 9]
]

# Text label parameters
labels = [
    "Treatment Effect",
    "No Improvement",
    "CBT",
    "Mindfulness",
    "Medication",
    "Control"
]
xlabel = "Therapy Sessions"
ylabel = "Stress Level"
ylim = [20, 80]

axhlines = [40, 60]  # Stress levels considered acceptable
title = "Effect of Different Therapies on Stress Levels"
supertitle = "Psychological Study Results"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

bp1 = ax.boxplot(
    data1,
    positions=np.array(range(len(data1))) * 2.0 - 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp2 = ax.boxplot(
    data2,
    positions=np.array(range(len(data2))) * 2.0,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp3 = ax.boxplot(
    data3,
    positions=np.array(range(len(data3))) * 2.0 + 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)

# Set properties for each boxplot
for bp, color in zip([bp1, bp2, bp3], ["#6a5acd", "#32cd32", "#ff4500"]):
    for patch in bp["boxes"]:
        patch.set_facecolor(color)
    for whisker in bp["whiskers"]:
        whisker.set(color="gray", linewidth=1)
    for cap in bp["caps"]:
        cap.set(color="gray", linewidth=1)
    for median in bp["medians"]:
        median.set(color="black", linewidth=2)

# Get the bottom and height of the boxes in bp2
box_data = [
    np.abs(box.get_path().vertices[1][1] - box.get_path().vertices[2][1])
    for box in bp2["boxes"]
]

max_box = np.max(box_data) 
min_box = np.min(box_data) 
max_pos = np.argmax(box_data)
min_pos = np.argmin(box_data)

ax.plot(
    max_pos * 2.0,
    bp2["medians"][max_pos].get_ydata()[0],
    marker="x",
    color="darkred",
    markersize=10,
)
ax.plot(
    min_pos * 2.0,
    bp2["medians"][min_pos].get_ydata()[0],
    marker="x",
    color="darkred",
    markersize=10,
)

# Add dashed lines for stress level thresholds
ax.axhline(y=axhlines[0], color="red", linestyle="--", label=labels[1])
ax.axhline(y=axhlines[1], color="red", linestyle="--", label=labels[1])

# Create Legend
axhline_legend = mlines.Line2D([], [], color="red", linestyle="--", label=labels[1])
marker_legend = mlines.Line2D([], [], color="darkred", marker="x", linestyle="None", label="Outliers")

patch1 = mpatches.Patch(color="#6a5acd", label=labels[2])
patch2 = mpatches.Patch(color="#32cd32", label=labels[3])
patch3 = mpatches.Patch(color="#ff4500", label=labels[4])

ax.legend(
    handles=[patch1, patch2, patch3, axhline_legend, marker_legend],
    loc="upper center",
    ncol=3,
    frameon=False,
)

ax.set_title(title, pad=20)
fig.suptitle(supertitle, fontsize=14, fontweight='bold')

ax.set_ylim(ylim)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

ax.set_xticks(range(0, len(data1) * 2, 2))
ax.set_xticklabels(["Pre-Therapy", "Post-Therapy (2 weeks)", "Post-Therapy (1 month)", "Post-Therapy (3 months)", "Post-Therapy (6 months)"])

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("box_38.pdf", bbox_inches="tight")
