
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Generate data for electromagnetic field simulation
x = np.linspace(-10, 10, 200)
y = np.linspace(-10, 10, 200)
X, Y = np.meshgrid(x, y)
Z_E = np.sin(np.sqrt(X**2 + Y**2)) * np.exp(-0.1*np.sqrt(X**2 + Y**2))
Z_B = np.cos(np.sqrt(X**2 + Y**2)) * np.exp(-0.1*np.sqrt(X**2 + Y**2))
labels = ["E-Field", "B-Field"]
xlabel = "X-axis (meters)"
ylabel = "Y-axis (meters)"
title = "Electromagnetic Field Distribution"
cbar_label = "Field Intensity (arbitrary units)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the plot
fig, ax = plt.subplots(figsize=(12, 9))

# Contour lines for E-Field (blue shades) and B-Field (orange shades)
CS_E = ax.contour(X, Y, Z_E, colors="blue", linestyles="solid", linewidths=1.5)
CS_B = ax.contour(X, Y, Z_B, colors="orange", linestyles="dashed", linewidths=1.5)

# Labels for x and y axes
ax.set_xlabel(xlabel, fontsize=15)
ax.set_ylabel(ylabel, fontsize=15)
ax.set_title(title, fontsize=18)

# Adding a legend manually
h1, _ = CS_E.legend_elements()
h2, _ = CS_B.legend_elements()
ax.legend([h1[0], h2[0]], labels, fontsize=13)

# Set the aspect of the plot
ax.set_aspect("equal")
ax.grid(True, linestyle=':', color='gray')
ax.set_facecolor("#e0f7fa")
ax.set_ylim(-10, 10)
ax.set_xlim(-10, 10)

# Add color bar to represent field intensity
field_intensity = ax.contourf(X, Y, Z_E + Z_B, alpha=0.3, cmap="inferno")
cbar = fig.colorbar(field_intensity, ax=ax)
cbar.set_label(cbar_label, fontsize=13)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("contour_63.pdf", bbox_inches="tight")
