# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)  # Set a seed for reproducibility

from scipy.stats import gaussian_kde
# Generate data for the plot
x = np.linspace(0, 24, 1000)  # Time in months, over 2 years
# Simulated data for three different journal types with complex variations
y_data = [
    np.abs(np.random.uniform(0.6, 1.4) * np.sin(0.1 * x * (index + 1))
           * np.exp(-0.03 * (x - i) ** 2 / np.linspace(0.3, 0.9, 3)[index])
           + np.random.normal(0, 0.05, x.shape))  # Add some noise for complexity
    for index, i in enumerate(np.linspace(1, 15, 3))
]

# Combine all journal data into a single array for density estimation
all_data = np.concatenate(y_data)

# Extracted variables
xlabel_text = "Submission / Acceptance Rate"
ylabel_text = "Density"
title_text = "Density of Manuscript Submissions and Acceptance Rates Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Colors for different density curves
colors = plt.cm.viridis(np.linspace(0, 1, len(y_data)))

# Plot separate density curves for each journal type
for i, y in enumerate(y_data):
    kde = gaussian_kde(y, bw_method='scott')
    x_dens = np.linspace(min(y), max(y), 1000)
    density = kde(x_dens)
    ax.plot(x_dens, density, color=colors[i], lw=2, label=f"Journal Type {i+1}")
    ax.fill_between(x_dens, density, color=colors[i], alpha=0.2)

# Set labels and title
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(True)
ax.spines["left"].set_visible(True)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)

# Add legend
plt.legend(title="Journal Types", loc='upper right', fontsize=10)

# Add grid for readability
plt.grid(True, linestyle=":", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_40.pdf", bbox_inches="tight")
