# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation - Annual Income Data
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
occupations = ['Engineer', 'Doctor', 'Teacher', 'Lawyer', 'Artist', 'Scientist', 'Entrepreneur']
# Simulated annual income data for each occupation (in thousands)
incomes = [
    np.random.normal(60 + i*10, 15, 300)  # Mean income varies by occupation
    for i in range(len(occupations))
]

# Adding one more set of income data for variation
additional_income = np.random.normal(90, 20, 300)  # A different set with a higher mean and variance

# Combine data for density estimation
income_data = np.concatenate(incomes + [additional_income])

# Kernel Density Estimation (KDE) for the income data
kde_income = gaussian_kde(income_data, bw_method='scott')
x_dens_income = np.linspace(min(income_data), max(income_data), 1000)
density_income = kde_income(x_dens_income)

# Create density plot for the additional income data
kde_additional = gaussian_kde(additional_income, bw_method='scott')
density_additional = kde_additional(x_dens_income)

# Extracted variables for the density plot
xlabel_text = "Annual Income (Thousands)"
ylabel_text = "Density"
title_text = "Density of Annual Incomes with Additional Income Distribution"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Plot the density curve for overall income data
ax.plot(x_dens_income, density_income, color='green', lw=2, label='Overall Income Density')
ax.fill_between(x_dens_income, density_income, color='green', alpha=0.3)

# Plot the density area for additional income data
ax.fill_between(x_dens_income, density_additional, color='orange', alpha=0.5, label='Additional Income Density')

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right', fontsize=10)

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_44.pdf", bbox_inches="tight")
