# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation - Technology Adoption Data
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(2)
years = np.linspace(2010, 2020, 100)  # Simulate years from 2010 to 2020
# Simulated adoption rate data for 3 different technology trends
adoption_rates = [
    50 * np.sin(0.5 * (years - 2010) + i) + 50 + np.random.uniform(-10, 10, 100)
    for i in range(3)
]

# Combine data for density estimation
adoption_data_1 = adoption_rates[0]
adoption_data_2 = adoption_rates[1]

# Kernel Density Estimation (KDE) for the adoption rate data
kde_adoption_1 = gaussian_kde(adoption_data_1, bw_method='scott')
kde_adoption_2 = gaussian_kde(adoption_data_2, bw_method='scott')

x_dens_adoption = np.linspace(0, 100, 1000)
density_adoption_1 = kde_adoption_1(x_dens_adoption)
density_adoption_2 = kde_adoption_2(x_dens_adoption)

# Extracted variables for the density plot
xlabel_text = "Adoption Rate (%)"
ylabel_text = "Density"
title_text = "Density of Technology Adoption Trends (2010-2020)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density curves for adoption rate data
ax.plot(x_dens_adoption, density_adoption_1, color='blue', lw=2, label='Technology 1 Density')
ax.fill_between(x_dens_adoption, density_adoption_1, color='blue', alpha=0.3)

ax.plot(x_dens_adoption, density_adoption_2, color='green', lw=2, label='Technology 2 Density')
ax.fill_between(x_dens_adoption, density_adoption_2, color='green', alpha=0.3)

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right')

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_45.pdf", bbox_inches="tight")
