# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation - Energy Consumption Data
# ===================
import numpy as np
from scipy.stats import gaussian_kde
np.random.seed(42)  # For reproducibility
time = np.linspace(0, 24, 100)
solar_energy = 50 * np.maximum(0, np.sin(np.pi * time / 12) * (1 - np.abs(time - 12) / 24))
wind_energy = 30 + 10 * np.cos(np.pi * time / 6)

# Combine data for density estimation
energy_data = np.concatenate((solar_energy, wind_energy))

# Kernel Density Estimation (KDE) for the energy data
kde_energy = gaussian_kde(energy_data, bw_method='scott')
x_dens_energy = np.linspace(min(energy_data), max(energy_data), 1000)
density_energy = kde_energy(x_dens_energy)

# Extracted variables for the density plot
xlabel_text = "Energy Consumption (kWh)"
ylabel_text = "Density"
title_text = "Density of Energy Consumption from Solar and Wind over 24 Hours"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 5))

# Plot the density curve for energy consumption
ax.plot(x_dens_energy, density_energy, color='green', lw=2, label='Energy Density')
ax.fill_between(x_dens_energy, density_energy, color='green', alpha=0.3)

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right')

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_54.pdf", bbox_inches="tight")
