
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Fix a seed for reproducibility
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
# Generate bimodal distribution for GDP growth rates data
# Simulate GDP growth rates for two countries (Country A and Country B)
gdp_growth_country_a = np.random.normal(loc=2, scale=1, size=400)  # Stable growth rates
gdp_growth_country_b = np.random.normal(loc=5, scale=2, size=400)  # Higher growth rates
gdp_growth_data = np.concatenate([gdp_growth_country_a, gdp_growth_country_b])

# X-axis values for KDE
xs = np.linspace(-2, 10, 200)

# Axes Limits and Labels
title = "KDE Plot of GDP Growth Rates for Two Countries"
xlabel_value = "GDP Growth Rate (%)"
ylabel_value = "Density"
legend_label = "GDP Growth Density for Countries A and B"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xticks_values = [-2, 0, 2, 4, 6, 8, 10]
xticklabels = ["-2%", "0%", "2%", "4%", "6%", "8%", "10%"]
yticks_values = [0, 0.1, 0.2, 0.3, 0.4]
yticklabels = ["0", "0.1", "0.2", "0.3", "0.4"]
xlim_values = [-2, 10]
ylim_values = [0, 0.45]
title_fontsize = 16
title_weight = "bold"
label_fontsize = 14
legend_loc = "upper right"
legend_fontsize = 14

# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(gdp_growth_data)
density.covariance_factor = lambda: 0.25
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#99c2ff", edgecolor="navy", alpha=0.7)

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=label_fontsize)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=label_fontsize)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_loc, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_60.pdf", bbox_inches="tight")
