
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
# Fix a seed for reproducibility
import numpy as np

np.random.seed(42)

# Generate synthetic data for forest cover change in two regions
# Region A: Moderate changes
forest_change_region_a = np.random.normal(loc=0, scale=10, size=400)
# Region B: Drastic changes
forest_change_region_b = np.random.normal(loc=50, scale=20, size=400)

# Combine the data
forest_change_data = np.concatenate([forest_change_region_a, forest_change_region_b])

# X-axis values for KDE
xs = np.linspace(-50, 100, 200)

# Axes Limits and Labels
title = "KDE Plot of Forest Cover Change Distribution for Two Regions"
xlabel_value = "Forest Cover Change (%)"
ylabel_value = "Density"
legend_label = "Density of Forest Cover Change for Regions A and B"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xticks_values = [-50, 0, 50, 100]
xticklabels = ["-50", "0", "50", "100"]
yticks_values = [0, 0.005, 0.01, 0.015, 0.02]
yticklabels = ["0", "0.005", "0.01", "0.015", "0.02"]
xlim_values = [-50, 100]
ylim_values = [0, 0.022]
title_fontsize = 14
title_weight = "bold"
label_fontsize = 12
legend_loc = "upper right"
legend_fontsize = 12

# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(forest_change_data)
density.covariance_factor = lambda: 0.3
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#93c572", edgecolor="#6b8e23", alpha=0.6)

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=label_fontsize)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=label_fontsize)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_loc, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_61.pdf", bbox_inches="tight")
