
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Fix a seed for reproducibility
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)

# Generate bimodal distribution for movie ratings
# Simulate movie ratings for two genres (Action and Drama)
ratings_action = np.random.normal(loc=6, scale=1.5, size=400)  # Action movies
ratings_drama = np.random.normal(loc=7.5, scale=0.8, size=400)  # Drama movies
ratings_data = np.concatenate([ratings_action, ratings_drama])

# X-axis values for KDE
xs = np.linspace(0, 10, 200)

# Axes Limits and Labels
title = "KDE Plot of Movie Ratings Distribution for Two Genres"
xlabel_value = "Rating"
ylabel_value = "Density"
legend_label = "Rating Density for Action and Drama Movies"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xticks_values = [0, 2, 4, 6, 8, 10]
xticklabels = ["0", "2", "4", "6", "8", "10"]
yticks_values = [0, 0.1, 0.2, 0.3, 0.4]
yticklabels = ["0", "0.1", "0.2", "0.3", "0.4"]
xlim_values = [0, 10]
ylim_values = [0, 0.45]
title_fontsize = 14
title_weight = "bold"
label_fontsize = 12
legend_loc = "upper right"
legend_fontsize = 12
# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(ratings_data)
density.covariance_factor = lambda: 0.3
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#66c2a5", edgecolor="#fc8d62")

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=label_fontsize)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=label_fontsize)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_loc, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_62.pdf", bbox_inches="tight")
