
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(24)

# Generate sample tourism data
spring_tourists = np.random.normal(loc=200, scale=30, size=1000)  # in thousands
summer_tourists = np.random.normal(loc=450, scale=50, size=1000)  # in thousands
winter_tourists = np.random.normal(loc=150, scale=40, size=1000)  # in thousands

labels = ["Spring Tourists", "Summer Tourists", "Winter Tourists"]
avxlabel = "Average Spring Tourists"
xlabel = "Tourist Arrivals (in thousands)"
ylabel = "Density"
title = "Seasonal Tourist Arrivals Density Distributions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_linestyle = "-"
grid_linewidth = 0.5
grid_alpha = 0.7
line_linestyle = "--"
line_linewidth = 2
fill_alpha = 0.25

fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density plots
for data, color, label in zip(
    [spring_tourists, summer_tourists, winter_tourists],
    ["#77DD77", "#AEC6CF", "#CFCFC4"],  # Adapted specific colors for Spring, Summer, and Winter
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 20, max(data) + 20, 200)
    density.covariance_factor = lambda: 0.25  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_linestyle,  # dashed line
        linewidth=line_linewidth,  # thicker lines
    )
    plt.fill_between(
        xs, density(xs), color=color, alpha=fill_alpha
    )  # fill for shade

# Plot a vertical line indicating the average spring tourists
plt.axvline(
    x=np.mean(spring_tourists),
    color="red",
    linestyle=line_linestyle,
    linewidth=line_linewidth,
    label=avxlabel,
)

# Set labels and title
ax.set_xlim(
    min(min(spring_tourists), min(summer_tourists), min(winter_tourists)) - 20,
    max(max(spring_tourists), max(summer_tourists), max(winter_tourists)) + 20,
)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid with a subtle linestyle
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_68.pdf", bbox_inches="tight")
