
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

# Generate synthetic temperature data for different regions
np.random.seed(123)

# Coastal region, mean around 20°C, higher variance due to weather variability
temps_coastal = np.random.normal(loc=20, scale=3, size=1000)

# Inland region, mean around 25°C, moderate variance
temps_inland = np.random.normal(loc=25, scale=2, size=1000)

# Mountainous region, mean around 15°C, lower variance
temps_mountainous = np.random.normal(loc=15, scale=1.5, size=1000)

# Labels for the new data
labels = ["Coastal Region", "Inland Region", "Mountainous Region"]
avxlabel = "Overall Average Temperature"
xlabel = "Monthly Average Temperature (°C)"
ylabel = "Density"
title = "Monthly Average Temperature Distribution by Region"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure and axis
fig, ax = plt.subplots(figsize=(9, 6))

# Define new colors for the different regions
colors = ["#1f77b4", "#2ca02c", "#ff7f0e"]  # blue, green, orange

# Plot the density plots
for data, color, label in zip([temps_coastal, temps_inland, temps_mountainous], colors, labels):
    density = gaussian_kde(data)
    xs = np.linspace(5, 35, 200)
    density.covariance_factor = lambda: 0.4
    density._compute_covariance()
    plt.fill_between(xs, density(xs), color=color, alpha=0.3, label=label)

# Plot the overall average temperature line
overall_avg_temp = np.mean([temps_coastal.mean(), temps_inland.mean(), temps_mountainous.mean()])
plt.axvline(x=overall_avg_temp, color="purple", linestyle="--", label=avxlabel)

# Set chart labels and title
ax.set_xlim(5, 35)
ax.set_xticks(np.arange(5, 36, 5))
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
plt.title(title)

# Show grid
plt.grid(True, linestyle="--", alpha=0.6)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_71.pdf", bbox_inches="tight")
