
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for geographical factors influencing urban air quality
categories = [
    "Industrial Emissions",
    "Traffic Emissions",
    "Construction Activities",
    "Residential Heating",
    "Agricultural Activities",
    "Natural Sources",
    "Waste Management",
    "Public Transport",
    "Green Spaces",
    "Renewable Energy",
]
values = [0.30, 0.25, 0.20, 0.15, 0.10, 0.05, -0.05, -0.10, -0.15, -0.20]
errors = [0.06, 0.05, 0.04, 0.03, 0.03, 0.02, 0.02, 0.02, 0.01, 0.01]

min_val = min(values) - 0.1
max_val = max(values) + 0.1

# Function to normalize values to a 0-1 range for color scaling
def normalize(value, min_val, max_val):
    return (value - min_val) / (max_val - min_val)

# Axes Limits and Labels
ylabel_value = "Geographical Factors"
xlabel_value = "Impact on Urban Air Quality (Δ to control)"
title_value = "Impact of Various Geographical Factors on Urban Air Quality"
suptitle_value = "An Analysis of Contributing Factors"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis

# Determine color based on normalized value
def get_color(value):
    norm_value = normalize(value, min_val, max_val)
    blue_base = np.array(mcolors.to_rgb("#1f77b4"))
    # Create a color that ranges from very light blue to the base blue
    return mcolors.to_hex((1 - blue_base) * (1 - norm_value) + blue_base)

colors = [get_color(value) for value in values]

fig, ax = plt.subplots(figsize=(10, 8))

# Horizontal bar chart
bars = ax.barh(
    categories, values, xerr=errors, color=colors, capsize=5, edgecolor="grey"
)
ax.set_ylabel(ylabel_value)
ax.set_xlabel(xlabel_value)
ax.set_title(title_value)
fig.suptitle(suptitle_value, fontsize=14, fontweight='bold')

# Set y-axis limits and x-axis limits
ax.set_xlim(min_val, max_val)  # Adjust limits to encompass errors

# Remove top and right spines for a cleaner look
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)

# Customize grid lines
ax.xaxis.grid(True, linestyle="--", which="major", color="grey", alpha=0.6)
ax.set_axisbelow(True)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to prevent clipping of ylabel
plt.tight_layout()
plt.savefig("errorbar_112.pdf", bbox_inches="tight")
