
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
# Data setup
import numpy as np

np.random.seed(42)

regions = ["Region A", "Region B", "Region C", "Region D"]
x = np.arange(len(regions))  # X-axis points

# Climate change metrics for different regions
carbon_emissions = np.array([10.2, 15.4, 12.8, 18.1])  # Simulated values
renewable_energy_usage = np.array([5.6, 6.1, 7.2, 8.5])  # Simulated values

# Errors for both metrics
carbon_emissions_errors = np.array([1.0, 1.5, 1.2, 1.4])
renewable_energy_usage_errors = np.array([0.8, 0.9, 1.0, 1.1])

labels = ["Carbon Emissions (MtCO2)", "Renewable Energy Usage (%)"]
ylabels = ["Carbon Emissions (Metric Tons of CO2)", "Renewable Energy Usage (%)"]

title = "Climate Change Metrics Across Regions"
ylims = [[0, 25], [0, 15]]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting setup
fig, ax1 = plt.subplots(figsize=(8, 5))

# Bar width and hatch patterns
width = 0.35
hatch_patterns = ["//", "o"]

# Colors for the bars
colors = ["#228B22", "#1E90FF"]  # Forest green and dodger blue

# Plot data on the left y-axis (carbon emissions)
ax1.bar(
    x,
    carbon_emissions,
    width,
    color=colors[0],
    hatch=hatch_patterns[0],
    label=labels[0],
    yerr=carbon_emissions_errors,
    capsize=5,
    edgecolor="black",
)

# Create a second y-axis sharing the same x-axis (renewable energy usage)
ax2 = ax1.twinx()
ax2.bar(
    x + width,
    renewable_energy_usage,
    width,
    color=colors[1],
    hatch=hatch_patterns[1],
    label=labels[1],
    yerr=renewable_energy_usage_errors,
    capsize=5,
    edgecolor="black",
)

# Set the x-ticks to be in the middle of the two bars and add the labels
ax1.set_xticks(x + width / 2)
ax1.set_xticklabels(regions)

# Add a legend
ax1.legend(loc="upper left")
ax2.legend(loc="upper right")

# Set labels for y-axes
ax1.set_ylabel(ylabels[0], color=colors[0])
ax2.set_ylabel(ylabels[1], color=colors[1])

# Set colors for y-axes
ax1.tick_params(axis="y", colors=colors[0])
ax2.tick_params(axis="y", colors=colors[1])

# Set the limits for y-axes to fit the data
ax1.set_ylim(ylims[0])
ax2.set_ylim(ylims[1])

# Set title for the chart
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorbar_120.pdf", bbox_inches="tight")
