
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
judges = ["Expert 1", "Expert 2", "Expert 3", "Expert 4"]
protocols = [
    "Social Media",
    "Email Marketing",
    "SEO",
    "Content Marketing",
    "Influencer Marketing",
    "Pay Per Click",
]
effectiveness_means = np.array(
    [
        [75, 50, 60, 70, 80, 90],  # Expert 1
        [70, 55, 65, 75, 85, 95],  # Expert 2
        [80, 60, 70, 80, 90, 100],  # Expert 3
        [85, 65, 75, 85, 95, 105],  # Expert 4
    ]
)
effectiveness_std = np.array(
    [
        [7, 7, 7, 7, 7, 7],  # Expert 1
        [8, 6, 8, 6, 8, 6],  # Expert 2
        [6, 8, 6, 8, 6, 8],  # Expert 3
        [7, 6, 8, 7, 6, 7],  # Expert 4
    ]
)
legendtitle = "Campaign Strategy"
xlabel = "Marketing Experts"
ylabel = "Effectiveness (%)"
ylim = [0, 120]

# Labels for plot
title = "Effectiveness of Different Marketing Campaign Strategies"
subtitle = "Evaluated by Different Marketing Experts"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set figure size to match the original image's dimensions
fig, ax = plt.subplots(figsize=(12, 7))

# colors
colors = [
    "#ff6f61",  # Social Media
    "#6b5b95",  # Email Marketing
    "#88b04b",  # SEO
    "#f7cac9",  # Content Marketing
    "#92a8d1",  # Influencer Marketing
    "#034f84",  # Pay Per Click
]

# Bar width
bar_width = 0.15
bar_width_main = 0.75

# Set position of bar on X axis
r = np.arange(len(judges))

# Draw bars for 'Social Media' protocol
i = protocols.index("Social Media")
ax.bar(
    r + (i + 3) * bar_width,
    effectiveness_means[:, i],
    yerr=effectiveness_std[:, i],
    width=bar_width_main,
    label=protocols[i],
    capsize=5,
    color=colors[i],
    hatch="//",
    edgecolor="black",
)

# Draw bars for other protocols
for i in range(len(protocols)):
    if protocols[i] == "Social Media":
        continue
    ax.bar(
        r + i * bar_width,
        effectiveness_means[:, i],
        yerr=effectiveness_std[:, i],
        width=bar_width,
        label=protocols[i],
        capsize=5,
        color=colors[i],
        edgecolor="black",
    )

# Add xticks on the middle of the group bars
ax.set_xlabel(xlabel)
ax.set_xticks(r + bar_width * (len(protocols) - 1) / 2)
ax.set_xticklabels(judges)

# Create title, subtitle, legend & Show graphic
ax.set_title(title, fontsize=16)
fig.suptitle(subtitle, y=1.02, fontsize=12)
handles, labels = ax.get_legend_handles_labels()
order = [0, 1, 2, 4, 3, 5]  # Reordering the legend
ax.legend(
    [handles[idx] for idx in order],
    [labels[idx] for idx in order],
    loc="upper center",
    bbox_to_anchor=(0.5, 1.15),
    ncol=6,
    title=legendtitle,
)
ax.set_ylabel(ylabel)
ax.set_ylim(ylim)  # Adjust y-axis limit to accommodate error bars

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorbar_34.pdf", bbox_inches="tight")
