
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Regions for psychological data
regions = ["Europe", "Asia", "North America", "South America", "Africa"]

# Average Happiness Score (scale 1-10)
happiness_score = [7.5, 6.0, 7.2, 6.5, 5.8]  # Approximate values
happiness_score_errors = [0.6, 0.7, 0.6, 0.4, 0.5]  # Error estimates

# Percentage of Population with Access to Mental Health Services
mental_health_access = [70, 50, 65, 45, 30]  # Approximate percentages
mental_health_access_errors = [5, 7, 6, 8, 7]  # Error estimates

# Percentage of Population Experiencing Stress
stress_percentage = [30, 40, 35, 25, 45]  # Approximate percentages
stress_percentage_errors = [4, 5, 4, 3, 6]  # Error estimates

titles = [
    "Average Happiness Score",
    "Access to Mental Health Services",
    "Percentage of Population Experiencing Stress",
]
xlabels = ["Happiness Score (1-10)", "% of Population", "% of Population"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a subplot for each category
fig, (ax1, ax2, ax3) = plt.subplots(3, 1, figsize=(6, 9))
# Custom colors for psychology charts
colors = ["#6baed6", "#9ecae1", "#c6dbef", "#fde0dd", "#fa9fb5"]

def add_bars_with_annotations(ax, data, errors, colors):
    bars = ax.barh(regions, data, xerr=errors, color=colors, capsize=5, edgecolor="black")
    for i, bar in enumerate(bars):
        width = bar.get_width()
        label_x_pos = bar.get_width() + errors[i] * 0.2
        ax.text(
            label_x_pos, bar.get_y() + bar.get_height(), f"{width:.1f}", va="center"
        )

# Plot Average Happiness Score on the first subplot
ax1.barh(
    regions,
    happiness_score,
    xerr=happiness_score_errors,
    edgecolor="black",
    color=colors,
    capsize=5,
)
add_bars_with_annotations(ax1, happiness_score, happiness_score_errors, colors)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.xaxis.grid(True, linestyle="--")
ax1.spines["right"].set_visible(False)
ax1.spines["top"].set_visible(False)

# Plot Mental Health Access on the second subplot
ax2.barh(
    regions,
    mental_health_access,
    xerr=mental_health_access_errors,
    edgecolor="black",
    color=colors,
    capsize=5,
)
add_bars_with_annotations(ax2, mental_health_access, mental_health_access_errors, colors)
ax2.set_title(titles[1])
ax2.set_xlabel(xlabels[1])
ax2.xaxis.grid(True, linestyle="--")
ax2.spines["right"].set_visible(False)
ax2.spines["top"].set_visible(False)

# Plot Stress Percentage on the third subplot
ax3.barh(
    regions,
    stress_percentage,
    xerr=stress_percentage_errors,
    edgecolor="black",
    color=colors,
    capsize=5,
)
add_bars_with_annotations(ax3, stress_percentage, stress_percentage_errors, colors)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[2])
ax3.xaxis.grid(True, linestyle="--")
ax3.spines["right"].set_visible(False)
ax3.spines["top"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("errorbar_82.pdf", bbox_inches="tight")
