
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Regions for technology data
regions = ["North America", "Europe", "Asia", "South America", "Africa"]

# Market Share of Technology Companies (%)
market_share = [45, 30, 50, 20, 15]  # Hypothetical values
market_share_errors = [4, 3, 5, 2, 2]  # Error estimates

# Innovation Index (hypothetical scores out of 100)
innovation_index = [90, 85, 88, 75, 70]  # Hypothetical scores
innovation_index_errors = [5, 4, 6, 4, 4]  # Error estimates

# R&D Expenditure (billions of USD)
rd_expenditure = [200, 150, 180, 50, 40]  # Hypothetical expenditures
rd_expenditure_errors = [20, 15, 18, 5, 4]  # Error estimates

titles = [
    "Market Share of Technology Companies",
    "Innovation Index",
    "R&D Expenditure",
]
xlabels = ["% Market Share", "Innovation Score", "Billions of USD"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a subplot for each category
fig, (ax1, ax2, ax3) = plt.subplots(3, 1, figsize=(8, 10))
# Colors for bars, each plot can have its own color map or shared
colors = plt.get_cmap("cool")(np.linspace(0.2, 0.8, 5))

def add_bars_with_annotations(ax, data, errors, colors):
    bars = ax.barh(regions, data, xerr=errors, color=colors, edgecolor="black", capsize=0)
    for i, bar in enumerate(bars):
        width = bar.get_width()
        label_x_pos = bar.get_width() + errors[i] * 0.2
        ax.text(label_x_pos, bar.get_y() + bar.get_height() / 2, f"{width}", va="bottom")

# Plot Market Share on the first subplot
ax1.barh(regions, market_share, xerr=market_share_errors, edgecolor="black", color=colors, capsize=0)
add_bars_with_annotations(ax1, market_share, market_share_errors, colors)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.xaxis.grid(True, linestyle="--")
ax1.spines["right"].set_visible(False)
ax1.spines["top"].set_visible(False)

# Plot Innovation Index on the second subplot
ax2.barh(regions, innovation_index, xerr=innovation_index_errors, edgecolor="black", color=colors, capsize=0)
add_bars_with_annotations(ax2, innovation_index, innovation_index_errors, colors)
ax2.set_title(titles[1])
ax2.set_xlabel(xlabels[1])
ax2.xaxis.grid(True, linestyle="--")
ax2.spines["right"].set_visible(False)
ax2.spines["top"].set_visible(False)

# Plot R&D Expenditure on the third subplot
ax3.barh(regions, rd_expenditure, xerr=rd_expenditure_errors, edgecolor="black", color=colors, capsize=0)
add_bars_with_annotations(ax3, rd_expenditure, rd_expenditure_errors, colors)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[2])
ax3.xaxis.grid(True, linestyle="--")
ax3.spines["right"].set_visible(False)
ax3.spines["top"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("errorbar_84.pdf", bbox_inches="tight")
