
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for plotting
import numpy as np

np.random.seed(2)
sectors = [
    "Energy Production",
    "Transportation",
    "Industry",
    "Residential",
    "Agriculture",
    "Commercial",
    "Forestry",
    "Waste Management",
    "Public Services",
    "Miscellaneous",
]
carbon_footprints = [
    10.5,  # Gigatonnes of CO2
    8.4,
    7.9,
    5.6,
    3.4,
    4.2,
    2.7,
    1.8,
    2.3,
    1.5,
]
errors = [
    [0.8, -0.6],  # Error margins in Gigatonnes of CO2
    [0.9, -0.7],
    [1.0, -0.8],
    [0.6, -0.5],
    [0.4, -0.3],
    [0.5, -0.4],
    [0.3, -0.2],
    [0.2, -0.15],
    [0.3, -0.2],
    [0.2, -0.15],
]
methods = [
    "Energy Calculations",
    "Transport Analysis",
    "Industrial Survey",
    "Residential Study",
    "Agricultural Data",
    "Commercial Estimation",
    "Forestry Reports",
    "Waste Data Models",
    "Public Services Review",
    "Miscellaneous Sources",
]
xticks = np.arange(0.0, 12.0, 2.0)  # Scale for carbon footprints
xlim = [0.0, 12.0]
xvline = 5.0  # Average carbon footprint in Gigatonnes of CO2
xvspan = [4.5, 5.5]  # Range near average
xlabel = "Carbon Footprint (Gigatonnes of CO2)"
ylabel = "Sectors"
title = "Carbon Footprints of Different Sectors"
supertitle = "An Overview of Global Carbon Emissions by Sector"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
fig, ax = plt.subplots(figsize=(12, 8))

# Color palette for the sectors
colors = plt.cm.viridis(np.linspace(0, 1, len(sectors)))

# Error bars with different positive and negative values
for i, (sector, value, error, color) in enumerate(zip(sectors, carbon_footprints, errors, colors)):
    ax.errorbar(
        value,
        i,
        xerr=[[abs(error[1])], [error[0]]],
        fmt="o",
        color=color,
        ecolor='black',
        capsize=3,
    )
    ax.text(
        value,
        i - 0.15,
        r"$%.1f^{+%.2f} _{-%.2f}$" % (value, error[0], abs(error[1])),
        va="center",
        ha="center",
        fontsize=9,
        color='black'
    )

# Highlighted region with adjusted color and alpha
ax.axvspan(xvspan[0], xvspan[1], color="yellow", alpha=0.3)

# Text for methods with adjusted font size
for i, method in enumerate(methods):
    ax.text(12.5, i, method, va="center", ha="left", fontsize=11)

# Set labels and title
ax.set_yticks(range(len(sectors)))
ax.set_yticklabels(sectors, fontsize=11)
ax.set_xlabel(xlabel, fontsize=12)
ax.set_xlim(xlim)
ax.set_ylim(-0.5, len(sectors) - 0.5)
ax.invert_yaxis()  # Invert y-axis to match the original image
ax.axvline(x=xvline, linestyle="--", color="red", linewidth=1)
# Adjust x-axis ticks and labels
ax.set_xticks(xticks)
ax.set_xticklabels([f"{x:.1f}" for x in xticks], fontsize=11)

# Set the super title and title
plt.suptitle(supertitle, fontsize=14, fontweight='bold')
plt.title(title, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save/show the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("errorpoint_57.pdf", bbox_inches="tight")
