
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for plotting
import numpy as np

np.random.seed(42)
cities = [
    "New York",
    "Los Angeles",
    "Chicago",
    "Houston",
    "Phoenix",
    "Philadelphia",
    "San Antonio",
    "San Diego",
    "Dallas",
    "San Jose",
]
temperature_changes = [
    0.5,  # Degree Celsius
    0.8,
    0.6,
    1.0,
    0.4,
    0.7,
    0.9,
    0.5,
    0.6,
    0.4,
]
errors = [
    [0.2, -0.1],  # Error margins in Degree Celsius
    [0.3, -0.2],
    [0.1, -0.1],
    [0.2, -0.2],
    [0.1, -0.1],
    [0.2, -0.1],
    [0.3, -0.2],
    [0.1, -0.1],
    [0.2, -0.1],
    [0.1, -0.1],
]
methods = [
    "Historical Data Analysis",
    "Satellite Observations",
    "Weather Stations",
    "Climate Models",
    "Urban Heat Island Effect",
    "Long-term Weather Forecasts",
    "Meteorological Studies",
    "Ground Temperature Sensors",
    "Seasonal Analysis",
    "Oceanic Influence",
]
xticks = np.arange(0.0, 1.4, 0.2)  # Adjusted for smaller scale
xlim = [0.0, 1.4]
xvline = 0.7  # Average temperature change in Degree Celsius
xvspan = [0.6, 0.8]  # Range near average temperature change
xlabel = "Temperature Change (°C)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
fig, ax = plt.subplots(figsize=(10, 8))

# Error bars with different positive and negative values
for i, (city, value, error) in enumerate(zip(cities, temperature_changes, errors)):
    ax.errorbar(
        value,
        i,
        xerr=[[abs(error[1])], [error[0]]],
        fmt="o",
        color="green",
        ecolor="blue",
        capsize=3,
    )
    ax.text(
        value,
        i - 0.15,
        r"$%.1f^{+%.2f} _{-%.2f}$" % (value, error[0], abs(error[1])),
        va="center",
        ha="center",
        fontsize=9,
    )

# Highlighted region with adjusted color and alpha
ax.axvspan(xvspan[0], xvspan[1], color="lightblue", alpha=0.3)

# Text for methods with adjusted font size
for i, method in enumerate(methods):
    ax.text(1.4, i, method, va="center", ha="left", fontsize=11)

# Set labels and title
ax.set_yticks(range(len(cities)))
ax.set_yticklabels(cities)
ax.set_xlabel(xlabel, fontsize=12)
ax.set_xlim(xlim)
ax.invert_yaxis()  # Invert y-axis to match the original image
ax.axvline(x=xvline, linestyle="--", color="orange")
# Adjust x-axis ticks and labels
ax.set_xticks(xticks)
ax.set_xticklabels([f"{x:.1f}" for x in xticks])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save/show the plot
plt.tight_layout()
plt.savefig("errorpoint_58.pdf", bbox_inches="tight")
