
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Data for plotting
np.random.seed(2)
politicians = [
    "John Doe",
    "Jane Smith",
    "Robert Johnson",
    "Emily Davis",
    "Michael Brown",
    "Sarah Wilson",
    "David Martinez",
    "Linda Lee",
    "James Garcia",
    "Elizabeth Miller",
]
approval_ratings = [
    58.2,
    72.5,
    65.8,
    49.4,
    55.3,
    68.7,
    61.2,
    46.8,
    73.4,
    52.1,
]
errors = [
    [4.2, -3.8],  # Error margins in Approval Rating (%)
    [3.5, -2.9],
    [4.7, -3.2],
    [4.1, -3.7],
    [3.2, -2.8],
    [4.3, -3.1],
    [3.6, -3.0],
    [3.9, -2.7],
    [4.0, -3.5],
    [3.8, -2.9],
]
methods = [
    "Public Polls",
    "Online Surveys",
    "Telephone Polls",
    "Focus Groups",
    "Expert Analysis",
    "Historical Data",
    "Statistical Models",
    "Media Analysis",
    "Social Media Metrics",
    "Voter Registration Data",
]
xticks = np.arange(40.0, 90.0, 5.0)  # Adjusted for the range of approval ratings
xlim = [40.0, 90.0]
xvline = 60.0  # Average approval rating in %
xvspan = [55.0, 65.0]  # Range near average approval rating
xlabel = "Approval Rating (%)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
fig, ax = plt.subplots(figsize=(10, 8))

# Error bars with different positive and negative values
for i, (politician, rating, error) in enumerate(zip(politicians, approval_ratings, errors)):
    ax.errorbar(
        rating,
        i,
        xerr=[[abs(error[1])], [error[0]]],
        fmt="o",
        color="black",
        ecolor="blue",
        capsize=5,
        markersize=8,
    )
    ax.text(
        rating,
        i - 0.15,
        r"$%.1f^{+%.2f} _{-%.2f}$" % (rating, error[0], abs(error[1])),
        va="center",
        ha="center",
        fontsize=9,
    )

# Highlighted region with customized color and alpha
ax.axvspan(xvspan[0], xvspan[1], color="lightgreen", alpha=0.3)

# Text for methods with adequate font size
for i, method in enumerate(methods):
    ax.text(92, i, method, va="center", ha="left", fontsize=11)

# Set labels and title
ax.set_yticks(range(len(politicians)))
ax.set_yticklabels(politicians)
ax.set_xlabel(xlabel, fontsize=12)
ax.set_xlim(xlim)
ax.invert_yaxis()  # Invert y-axis to match the original image
ax.axvline(x=xvline, linestyle="--", color="red", linewidth=1.5)

# Adjust x-axis ticks and labels
ax.set_xticks(xticks)
ax.set_xticklabels([f"{x:.1f}" for x in xticks])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save/show the plot
plt.tight_layout()
plt.savefig("errorpoint_59.pdf", bbox_inches="tight")
