
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Set random seed
np.random.seed(0)

# Define the types of renewable energy sources
energy_sources = ["Solar", "Wind", "Hydropower"]

# Define the x positions for the data points
x_positions = np.linspace(0, 20, 6)

# Generate random energy output data for each energy source
solar_energy_output = np.random.uniform(60, 120, 6)
wind_energy_output = np.random.uniform(40, 90, 6)
hydropower_energy_output = np.random.uniform(50, 100, 6)

# Generate random error values for each energy source
solar_error = [np.random.uniform(6, 12, 6), np.random.uniform(6, 12, 6)]
wind_error = [np.random.uniform(4, 8, 6), np.random.uniform(4, 8, 6)]
hydropower_error = np.random.uniform(5, 10, 6)

# Define plot line positions and titles
horizontal_line_position = 85
vertical_line_position = 10

# Titles for each subplot
titles = ["Solar Energy Output", "Wind Energy Output", "Hydropower Energy Output"]

# Axis labels and other texts
xlabel = "Measurement Points"
ylabel_solar = "Energy Output (kWh)"
ylabel_wind = "Energy Output (kWh)"
ylabel_hydropower = "Energy Output (kWh)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with three subplots and shared x-axis
fig, (ax0, ax1, ax2) = plt.subplots(figsize=(6, 9), nrows=3, sharex=True)

# Define color schemes for each energy source
solar_colors = plt.get_cmap("YlOrBr")(np.linspace(0.2, 0.8, 6))
wind_colors = plt.get_cmap("Blues")(np.linspace(0.2, 0.8, 6))
hydropower_color = "#228B22"

# First subplot: Solar Energy Output with symmetric vertical error bars
for i in range(len(x_positions)):
    ax0.errorbar(
        x_positions[i],
        solar_energy_output[i],
        yerr=[[solar_error[0][i]], [solar_error[1][i]]],
        fmt="o",
        color=solar_colors[i],
        capsize=4,
    )
    ax0.text(
        x_positions[i] - 0.5,
        solar_energy_output[i],
        f"{solar_energy_output[i]:.2f}",
        fontsize=8,
        ha="right",
    )
ax0.set_title(titles[0])
ax0.axhline(y=horizontal_line_position, linestyle="--", color="#FFA500")
ax0.set_ylabel(ylabel_solar)
ax0.yaxis.grid(True)
ax0.xaxis.grid(False)

# Second subplot: Wind Energy Output with symmetric horizontal error bars
for i in range(len(x_positions)):
    ax1.errorbar(
        x_positions[i],
        wind_energy_output[i],
        xerr=[[wind_error[0][i]], [wind_error[1][i]]],
        fmt="o",
        color=wind_colors[i],
        capsize=4,
    )
    ax1.text(
        x_positions[i] + 0.5,
        wind_energy_output[i] + 0.1,
        f"{wind_energy_output[i]:.2f}",
        fontsize=8,
        ha="left",
    )
ax1.set_title(titles[1])
ax1.axvline(x=vertical_line_position, linestyle="--", color="#1E90FF")
ax1.set_ylabel(ylabel_wind)
ax1.xaxis.grid(True)
ax1.yaxis.grid(False)

# Third subplot: Hydropower Energy Output with symmetric vertical error bars
ax2.errorbar(
    x_positions,
    hydropower_energy_output,
    yerr=hydropower_error,
    fmt="*",
    color=hydropower_color,
    capsize=4,
)
ax2.set_title(titles[2])
ax2.set_ylabel(ylabel_hydropower)
ax2.set_xlabel(xlabel)
ax2.legend([energy_sources[2]], loc="upper left")
ax2.yaxis.grid(True)
ax2.xaxis.grid(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("errorpoint_77.pdf", bbox_inches="tight")
