
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Import necessary library
import networkx as nx
import numpy as np

# Create a directed graph
G = nx.DiGraph()

# Define nodes with city-related attributes
cities = {
    0: ("New York", "blue"),        # Coastal
    1: ("Los Angeles", "green"),    # Coastal
    2: ("Chicago", "orange"),       # Inland
    3: ("Houston", "brown"),        # Coastal
    4: ("Phoenix", "purple"),       # Desert
    5: ("Philadelphia", "red"),     # Coastal
    6: ("San Francisco", "cyan"),   # Coastal
}

for node, (city, color) in cities.items():
    G.add_node(node, city=city, color=color)

# Add edges with labels representing routes
routes = [
    (0, 1, "Flight"),
    (1, 2, "Train"),
    (2, 3, "Road"),
    (3, 4, "Flight"),
    (4, 5, "Road"),
    (5, 6, "Flight"),
    (6, 0, "Train"),
]
for u, v, label in routes:
    G.add_edge(u, v, label=label)

# Define node positions using a spring layout for better visualization
pos = nx.spring_layout(G)

title = "Major City Communication Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw nodes with color attribute
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=800, ax=ax)

# Draw edges with various styles
nx.draw_networkx_edges(
    G,
    pos,
    arrows=True,
    arrowstyle="-|>",
    arrowsize=20,
    width=2,
    edge_color="black",
    ax=ax
)
edge_labels = {(u, v): G[u][v]["label"] for u, v in G.edges}
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=edge_labels, font_color="darkred", font_size=10, ax=ax
)

# Draw node labels indicating city names
node_labels = {node: G.nodes[node]["city"] for node in G.nodes}
nx.draw_networkx_labels(
    G, pos, labels=node_labels, font_size=12, font_color="black", ax=ax
)

# Remove axis and add title
plt.axis("off")
plt.title(title, fontsize=16, fontweight="bold")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_64.pdf", bbox_inches="tight")
