
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
# import numpy for future enhancements
import numpy as np

# Create a directed graph for an ecosystem
G = nx.DiGraph()

# Define nodes with species-related attributes
nodes = {
    0: ("Grass", "forestgreen"),
    1: ("Rabbit", "lightgray"),
    2: ("Fox", "chocolate"),
    3: ("Eagle", "gold"),
    4: ("Mouse", "brown"),
    5: ("Snake", "darkgreen"),
    6: ("Bacteria", "purple"),
}
for node, (species, color) in nodes.items():
    G.add_node(node, species=species, color=color)

# Add edges with labels representing interactions
edges = [
    (0, 1, "Eaten by"),
    (1, 2, "Eaten by"),
    (2, 3, "Eaten by"),
    (1, 4, "Eaten by"),
    (4, 5, "Eaten by"),
    (5, 6, "Decomposed by"),
    (6, 0, "Fertilizes"),
]
for u, v, label in edges:
    G.add_edge(u, v, label=label)

# Define node positions in a circular layout
pos = nx.circular_layout(G)

title = "Ecosystem Interaction Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw nodes with color attribute
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=1000, ax=ax)

# Draw edges with labels and custom styles
nx.draw_networkx_edges(
    G,
    pos,
    arrows=True,
    arrowstyle="-|>",
    arrowsize=20,
    width=2,
    edge_color="darkgray",
    style="dashed",
    ax=ax,
)
edge_labels = {(u, v): G[u][v]["label"] for u, v in G.edges}
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=edge_labels, font_color="darkred", font_size=10, ax=ax
)

# Draw node labels indicating species names
node_labels = {node: G.nodes[node]["species"] for node in G.nodes}
nx.draw_networkx_labels(
    G, pos, labels=node_labels, font_size=12, font_color="black", ax=ax
)

# Remove axis and add title
plt.axis("off")
plt.title(title, fontsize=18, fontweight="bold")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_65.pdf", bbox_inches="tight")
