
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective city names and colors
nodes = {
    0: ("City A", "forestgreen"),
    1: ("City B", "darkorange"),
    2: ("City C", "royalblue"),
    3: ("Hub A1", "limegreen"),
    4: ("Hub A2", "yellowgreen"),
    5: ("Hub B1", "sandybrown"),
    6: ("Hub C1", "skyblue"),
    7: ("Hub C2", "cornflowerblue"),
}
for node, (location, color) in nodes.items():
    G.add_node(node, location=location, color=color)

# Add edges with capacities representing transport routes
edges = [
    (0, 1, "150km"),
    (0, 2, "200km"),
    (1, 3, "50km"),
    (1, 4, "75km"),
    (2, 5, "100km"),
    (2, 6, "90km"),
    (2, 7, "120km"),
]
for u, v, distance in edges:
    G.add_edge(u, v, distance=distance)

# Set network title and legend
title = "Transportation Network Among Cities"
legendtitle = "City and Hub Nodes"

# Define node positions in a spring layout
pos = nx.spring_layout(G, seed=42)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw nodes with color attribute and label them
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=800)
nx.draw_networkx_labels(
    G, pos, labels={node: G.nodes[node]["location"] for node in G.nodes}, font_size=10
)

# Draw edges with capacity labels
nx.draw_networkx_edges(G, pos, edge_color='gray', arrowstyle="->", arrowsize=15, width=2)
edge_labels = {(u, v): G[u][v]["distance"] for u, v in G.edges}
nx.draw_networkx_edge_labels(G, pos, edge_labels=edge_labels, font_size=8)

# Add a title
plt.title(title, fontsize=20, fontweight='bold')

# Add legend manually
legend_labels = {color: location for _, (location, color) in nodes.items()}
for color in set(node_colors):
    ax.plot(
        [],
        [],
        color=color,
        label=legend_labels[color],
        marker="o",
        markersize=10,
        linestyle="",
    )

# Removing the axis
plt.axis("off")

# Adding legend to the plot
plt.legend(title=legendtitle, loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_70.pdf", bbox_inches="tight")
