
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
import numpy as np
import random

# Create a random graph with 12 nodes and a 0.3 probability for edge creation
random.seed(20)
np.random.seed(42)  # For reproducibility
G = nx.erdos_renyi_graph(15, 0.25)
trade_values = {edge: np.random.randint(10, 500) for edge in G.edges()}
nx.set_edge_attributes(G, trade_values, "trade_volume")

# Use a spring layout for node positions
pos = nx.spring_layout(G, seed=42)

# Labels for countries
country_labels = {i: f"Country {i}" for i in G.nodes()}

# Draw edge labels
trade_labels = nx.get_edge_attributes(G, "trade_volume")

title = "Global Trade Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(12, 10))

# Draw nodes with an economic color scheme
node_color = [plt.cm.Blues(np.random.rand()) for _ in G.nodes()]

nx.draw(
    G, pos, node_size=800, node_color=node_color, edge_color="grey", with_labels=False
)

# Draw the labels for the nodes
nx.draw_networkx_labels(
    G, pos, labels=country_labels, font_color="black", font_size=12, font_weight="bold"
)

# Draw the edges with varying widths based on trade volumes
trade_widths = [trade_values[edge] * 0.02 for edge in G.edges()]
nx.draw_networkx_edges(
    G, pos, width=trade_widths, edge_color="darkgreen"
)

# Draw edge labels with trade volumes
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=trade_labels, font_color="darkred", font_size=10
)

# Add a title and configure layout
plt.title(title, size=20)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
# Save the plot
plt.savefig("graph_80.pdf", bbox_inches="tight")

