
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
case_types = ["Civil Case", "Criminal Case", "Family Case", "Corporate Case", "IP Case", "Labor Case", "Tax Case"]
law_firms = [
    "Law Firm A",
    "Law Firm B",
    "Law Firm C",
    "Law Firm D",
    "Law Firm E",
    "Law Firm F",
    "Law Firm G",
]

cases_handled = np.array(
    [
        [15, 30, 25, 10, 12, 18, 22],
        [25, 15, 35, 20, 30, 10, 5],
        [20, 25, 15, 30, 10, 35, 7],
        [5, 10, 5, 40, 20, 10, 15],
        [10, 20, 10, 25, 30, 15, 10],
        [15, 10, 40, 20, 5, 25, 30],
        [5, 25, 10, 10, 40, 20, 15],
    ]
)

cbarlabels = ["cases handled", "monthly client visits", "correlation coeff."]
suptitle="Legal Case Analytics"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar

def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.2f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw,
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts

fig, ((ax, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(10, 8))

# Heatmap 1: Cases handled by different law firms
im, _ = heatmap(
    cases_handled, case_types, law_firms, ax=ax, cmap="Blues", cbarlabel=cbarlabels[0]
)
annotate_heatmap(im, valfmt="{x:.1f}", size=7)

# Heatmap 2: Random monthly client visits
data = np.random.randint(50, 200, size=(7, 7))
y = [f"Branch {i}" for i in range(1, 8)]
x = [f"Month {i}" for i in list("ABCDEFG")]
im, _ = heatmap(data, y, x, ax=ax2, vmin=50, cmap="Greens", cbarlabel=cbarlabels[1])
annotate_heatmap(im, valfmt="{x:d}", size=7, threshold=100, textcolors=("blue", "white"))

# Heatmap 3: Random data categorized using BoundaryNorm
data = np.random.randn(6, 6)
y = [f"Dept. {i}" for i in range(1, 7)]
x = [f"Quarter {i}" for i in range(1, 7)]

qrates = list("ABCDEFG")
norm = matplotlib.colors.BoundaryNorm(np.linspace(-3.5, 3.5, 8), 7)
fmt = matplotlib.ticker.FuncFormatter(lambda x, pos: qrates[::-1][norm(x)])

im, _ = heatmap(
    data,
    y,
    x,
    ax=ax3,
    cmap=mpl.colormaps["coolwarm"].resampled(7),
    norm=norm,
    cbar_kw=dict(ticks=np.arange(-3, 4), format=fmt),
    cbarlabel="Quality Rating",
)

annotate_heatmap(
    im, valfmt=fmt, size=9, fontweight="bold", threshold=-1, textcolors=("blue", "black")
)

# Heatmap 4: Correlation matrix of cases handled by law firms
corr_matrix = np.corrcoef(cases_handled)
im, _ = heatmap(
    corr_matrix,
    case_types,
    case_types,
    ax=ax4,
    cmap="bwr",
    vmin=-1,
    vmax=1,
    cbarlabel=cbarlabels[2],
)

def func(x, pos):
    return f"{x:.2f}".replace("0.", ".").replace("1.00", "")

annotate_heatmap(im, valfmt=matplotlib.ticker.FuncFormatter(func), size=7)

# ===================
# Part 4: Saving Output
# ===================
plt.suptitle(suptitle)
plt.tight_layout()
plt.savefig("heatmap_104.pdf", bbox_inches="tight")
