
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Sample business performance data: a 5x6 grid
data = np.array(
    [
        [15.3, 52.0, 40.9, 37.0, 23.4, 86.3],
        [29.2, 41.2, 20.6, 17.8, 31.3, 73.9],
        [35.8, 32.3, 23.9, 26.2, 45.5, 54.3],
        [46.0, 27.1, 12.5, 14.1, 28.1, 49.2],
        [55.7, 38.1, 22.6, 34.5, 48.6, 39.5],
    ]
)

# X and Y labels
x_labels = ["Revenue", "Profit", "Growth Rate", "Customer Satisfaction", "Market Share", "R&D Spend"]
y_labels = ["HR", "Finance", "Sales", "IT", "Operations"]
xlabel = "KPIs"
ylabel = "Departments"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(12, 9))
# Defining the colormap from green to red
cmap = plt.cm.RdYlGn_r
# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle with a fixed size
        circle = plt.Circle((j, i), 0.4, color=color)
        ax.add_artist(circle)

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f}%", ha="center", va="center", color="black", fontsize=12)

# Set labels for x and y axes
ax.set_xticks(range(len(x_labels)))
ax.set_xticklabels(x_labels, ha="center", fontsize=12)
ax.set_yticks(range(len(y_labels)))
ax.set_yticklabels(y_labels, va="center", fontsize=12)

# Adding titles for the axes
ax.set_xlabel(xlabel, fontsize=14, weight='bold')
ax.set_ylabel(ylabel, fontsize=14, weight='bold')
ax.set_title("Business Performance Metrics by Department", fontsize=16, weight='bold')

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[0.1, 1, 10, 100], orientation="vertical")
cbar.ax.set_yticklabels(["0.1", "1", "10", "100"])

# ===================
# Part 4: Saving Output
# ===================
# Save the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_58.pdf", bbox_inches="tight")
