
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Defining the colormap from gold to deep purple
cmap = plt.cm.plasma

# Sample data: a 5x6 grid, similar to the provided heatmap
data = np.array(
    [
        [23.5, 45.2, 16.2, 8.0, 55.4, 60.3],
        [35.1, 50.3, 17.6, 10.1, 48.3, 29.7],
        [28.9, 30.6, 14.7, 22.3, 36.5, 54.2],
        [42.0, 25.1, 10.0, 5.3, 65.1, 46.6],
        [20.7, 38.9, 12.4, 18.5, 42.6, 53.0],
    ]
)

# X and Y labels
x_labels = ["Priest", "Imam", "Rabbi", "Monk", "Saint", "Scholar"]
y_labels = ["Priest", "Imam", "Rabbi", "Monk", "Saint"]
xlabel = "Interacted Role"
ylabel = "Interacting Role"
title = "Interaction Heatmap Among Various Religious Roles"
cbar_label = "Interaction Percentage"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle with a fixed size
        circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
        ax.add_artist(circle)

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f}%", ha="center", va="center", color="black")

# Set labels for x and y axes
ax.set_xticks(range(len(x_labels)))
ax.set_xticklabels(x_labels, ha="center")
ax.set_yticks(range(len(y_labels)))
ax.set_yticklabels(y_labels, va="center")

# Adding titles for the axes
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)
ax.set_title(title, fontsize=16)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[0.1, 1, 10, 100], orientation="vertical")
cbar.ax.set_yticklabels(["0.1", "1", "10", "100"])
cbar.set_label(cbar_label, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Save the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_59.pdf", bbox_inches="tight")
