
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
from matplotlib.colors import Normalize

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Sample data: Monthly average temperatures (Celsius) for 5 cities
# Rows: Cities (City A, City B, City C, City D, City E)
# Columns: Months (Jan, Feb, Mar, Apr, May, Jun, Jul, Aug, Sep, Oct, Nov, Dec)
data = np.array(
    [
        [-2.0, 0.1, 5.3, 12.1, 16.4, 20.2, 22.6, 21.9, 17.7, 11.2, 4.3, 0.3],
        [3.2, 5.3, 10.2, 15.5, 19.8, 23.2, 25.9, 25.5, 20.8, 14.6, 8.5, 3.7],
        [7.9, 9.4, 13.2, 17.6, 21.4, 25.0, 27.7, 27.1, 23.0, 17.1, 11.2, 8.1],
        [10.2, 12.1, 16.3, 20.7, 24.5, 27.8, 29.3, 28.8, 25.0, 19.2, 13.4, 10.9],
        [15.3, 17.6, 22.1, 25.6, 28.9, 32.0, 34.1, 33.5, 29.7, 23.8, 17.9, 14.2],
    ]
)

# X and Y labels
x_labels = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]
y_labels = ["City A", "City B", "City C", "City D", "City E"]
xlabel = "Month"
ylabel = "City"
title = "Average Monthly Temperatures (Celsius)"
cbar_label = "Temperature (°C)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(12, 8))

# Create a custom colormap
cmap = plt.cm.get_cmap("coolwarm")

# Set up the colormap and norm for temperatures
norm = Normalize(vmin=np.min(data), vmax=np.max(data))

# Create the scatter plot with adjusted circle sizes
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        color = cmap(norm(data[i, j]))
        circle = plt.Circle((j, i), 0.4, color=color)
        ax.add_artist(circle)
        ax.text(j, i, f"{data[i, j]:.1f}°C", ha="center", va="center", color="black")

# Set labels for x and y axes
ax.set_xticks(range(len(x_labels)))
ax.set_xticklabels(x_labels, rotation=45, ha="right")
ax.set_yticks(range(len(y_labels)))
ax.set_yticklabels(y_labels)

# Adding titles for the axes
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)

# Set the limits of the axes
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect('equal')
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, orientation="vertical")
cbar.set_label(cbar_label, fontsize=12)

# Set the title of the plot
plt.title(title, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
# Save the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_60.pdf", bbox_inches="tight")
