
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import pandas as pd

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generating a similar style data for demonstration
# Assuming there are 3 months and 3 regions
months = ["January", "February", "March"]
regions = ["Region 1", "Region 2", "Region 3"]
temperature_data = []

# Generating random temperatures and percentage changes for the example
for region in regions:
    for month in months:
        temperatures = np.random.randint(-10, 40, size=9)
        np.random.shuffle(temperatures)
        percent_change = np.round(np.random.uniform(-10, 10), 2)
        for rank, temp in enumerate(temperatures, start=1):
            temperature_data.append(
                {
                    "Region": region,
                    "Month": month,
                    "Rank": rank,
                    "Temperature": temp,
                    "PercentChange": percent_change,
                }
            )

# Create the DataFrame
df = pd.DataFrame(temperature_data)

# Pivoting the dataframe for the heatmap
df_pivot = df.pivot_table(
    index="Rank", columns=["Region", "Month"], values="Temperature", aggfunc="first"
)

title = "Geographical Temperature Analysis Heatmap"
xlabel = "Region and Month"
ylabel = "Rank"
color_bar_label = "Temperature (°C)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Initialize the matplotlib figure
fig, ax = plt.subplots(figsize=(10, 10))

# Create a custom colormap for the heatmap
cmap = plt.get_cmap("coolwarm")

# Draw the heatmap
cax = ax.imshow(df_pivot, cmap=cmap)

# Decorations
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set x-axis and y-axis labels
ax.set_xticks(range(len(df_pivot.columns)))
ax.set_xticklabels(
    [f"{col[0]} {col[1]}" for col in df_pivot.columns], rotation=45, ha="right"
)
ax.set_yticks(range(len(df_pivot.index)))
ax.set_yticklabels(df_pivot.index)

# Add annotations
for i in range(len(df_pivot.index)):
    for j in range(len(df_pivot.columns)):
        text_color = "white" if df_pivot.iloc[i, j] < 15 else "black"
        ax.text(j, i, df_pivot.iloc[i, j], ha="center", va="center", color=text_color)

# Add colorbar to be smaller
cbar = fig.colorbar(cax, ax=ax, fraction=0.1, pad=0.04)
cbar.set_label(color_bar_label)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_76.pdf", bbox_inches="tight")
