
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Defining a more vibrant colormap
cmap = plt.cm.viridis

# Data for the two subplots representing political voting patterns
data1 = np.array(
    [
        [34.2, 18.7, 2.8, 21.5, 14.8],
        [28.4, 22.1, 3.1, 18.9, 27.5],
        [37.0, 14.4, 1.6, 40.5, 6.5],
        [42.5, 10.3, 3.7, 25.1, 18.4],
        [33.8, 15.5, 5.2, 12.6, 32.9],
    ]
)

data2 = np.array(
    [
        [45.3, 3.8, 22.1, 13.9, 14.9],
        [22.7, 17.5, 19.0, 32.6, 8.2],
        [30.4, 5.9, 2.3, 52.4, 9.0],
        [28.6, 6.7, 12.4, 38.3, 14.0],
        [20.5, 22.5, 25.7, 11.5, 19.8],
    ]
)

# X and Y labels
x_labels = ["Liberal", "Conservative", "Independent", "Socialist", "Libertarian"]
y_labels = ["Liberal", "Conservative", "Independent", "Socialist", "Libertarian"]

# Subplot titles
titles = ["(a) Voting Distribution in Election 1", "(b) Voting Distribution in Election 2"]

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10, 100]
yticklabels = ["0.1", "1", "10", "100"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)


# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 25 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}%", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_86.pdf", bbox_inches="tight")

