
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Updated steps
time_steps = np.linspace(0, 200, 50)

# Generating distinct financial trends for each line
stock_price = 1.5 + 0.05 * time_steps  # Linear growth
market_cap = np.log(time_steps + 1) + 2.0  # Logarithmic growth
daily_returns = np.random.normal(loc=0, scale=0.5, size=len(time_steps))*40  # Random noise
compound_interest = np.exp(0.05 * time_steps / 100) * 100  # Exponential growth

# Simulating standard deviations for error
std_dev = 0.1
stock_price_std = np.full_like(stock_price, std_dev)
market_cap_std = np.full_like(market_cap, std_dev)
daily_returns_std = np.full_like(daily_returns, std_dev)
compound_interest_std = np.full_like(compound_interest, std_dev)

# Axes Limits and Labels
xlabel_value = "Time (months)"
xlim_values = [0, 200]
xticks_values = np.linspace(0, 200, 9)

ylabel_value_1 = "Value"
ylabel_value_2 = "Exponential Scale"
yticks_values_1 = np.arange(0, 10, 2)
yticks_values_2 = np.arange(0, 130, 10)
ylim_values_1 = [0, 10]
ylim_values_2 = [0, 120]

# Labels
label_stock = "Stock Price"
label_market_cap = "Market Cap"
label_daily_return = "Daily Returns"
label_compound_interest = "Compound Interest"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(6, 8))

# Plotting on the first subplot
ax1.plot(time_steps, stock_price, "o-", color="#1f77b4", label=label_stock)
ax1.fill_between(
    time_steps,
    stock_price - stock_price_std,
    stock_price + stock_price_std,
    color="#1f77b4",
    alpha=0.2,
)

ax1.plot(time_steps, market_cap, "s-", color="#ff7f0e", label=label_market_cap)
ax1.fill_between(
    time_steps,
    market_cap - market_cap_std,
    market_cap + market_cap_std,
    color="#ff7f0e",
    alpha=0.2,
)

# Plotting on the second subplot
ax2.plot(time_steps, daily_returns, "^--", color="#2ca02c", label=label_daily_return)
ax2.fill_between(
    time_steps,
    daily_returns - daily_returns_std,
    daily_returns + daily_returns_std,
    color="#2ca02c",
    alpha=0.2,
)

ax2.plot(
    time_steps,
    compound_interest,
    "*-",
    color="#d62728",
    label=label_compound_interest,
)
ax2.fill_between(
    time_steps,
    compound_interest - compound_interest_std,
    compound_interest + compound_interest_std,
    color="#d62728",
    alpha=0.2,
)

# Customizing the first subplot
ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax1.set_facecolor("#f9f9f9")

# Customizing the second subplot
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax2.set_facecolor("#f9f9f9")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_192.pdf", bbox_inches="tight")
