
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Updated steps
time_steps = np.linspace(0, 100, 50)

# Generating plausible environmental science data
temperature_values = 10 + np.sin(time_steps * 0.1) * 10  # Temperature trend
co2_values = np.linspace(300, 400, 50) + np.random.normal(scale=5, size=50)  # CO2 levels
ice_coverage_values = np.maximum(0, 100 - (time_steps * 0.5) + np.random.normal(scale=5, size=50))  # Ice loss
sea_level_values = np.maximum(0, 0.1 * time_steps + np.random.normal(scale=0.2, size=50))  # Sea level rise

# Simulating standard deviations for error
std_dev = 2
temperature_std = np.full_like(temperature_values, std_dev)
co2_std = np.full_like(co2_values, std_dev)
ice_coverage_std = np.full_like(ice_coverage_values, std_dev)
sea_level_std = np.full_like(sea_level_values, std_dev)

# Axes Limits and Labels
xlabel_value = "Time Steps (Years)"
xlim_values = [0, 100]
xticks_values = np.linspace(0, 100, 11)

ylabel_value_1 = "Temperature (°C)"
ylabel_value_2 = "Sea Level (m)"
yticks_values_1 = np.arange(0, 31, 5)
yticks_values_2 = np.arange(0, 11, 1)
ylim_values_1 = [0, 30]
ylim_values_2 = [0, 10]

# Labels
temperature_label = "Temperature (°C)"
co2_label = "CO2 Levels (ppm)"
ice_coverage_label = "Ice Coverage (km²)"
sea_level_label = "Sea Level (m)"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(8, 10))

# Plotting on the first subplot
ax1.plot(time_steps, temperature_values, "o-", color="#ff5733", label=temperature_label)
ax1.fill_between(
    time_steps,
    temperature_values - temperature_std,
    temperature_values + temperature_std,
    color="#ff5733",
    alpha=0.2,
)

ax1.plot(time_steps, co2_values, "s-", color="#33cc33", label=co2_label)
ax1.fill_between(
    time_steps,
    co2_values - co2_std,
    co2_values + co2_std,
    color="#33cc33",
    alpha=0.2,
)

ax2.plot(
    time_steps,
    ice_coverage_values,
    "^--",
    color="#3366ff",
    markerfacecolor="white",
    markeredgecolor="#3366ff",
    label=ice_coverage_label,
)
ax2.fill_between(
    time_steps,
    ice_coverage_values - ice_coverage_std,
    ice_coverage_values + ice_coverage_std,
    color="#3366ff",
    alpha=0.2,
)

ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax1.set_facecolor("#f9f9f9")

# Plotting on the second subplot
ax2.plot(
    time_steps, sea_level_values, "*-", color="#e67300", label=sea_level_label
)
ax2.fill_between(
    time_steps,
    sea_level_values - sea_level_std,
    sea_level_values + sea_level_std,
    color="#e67300",
    alpha=0.2,
)
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax2.set_facecolor("#f9f9f9")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_193.pdf", bbox_inches="tight")
