
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# New dataset: Technology domain
weeks = list(range(1, 29))
productivity_data = np.random.normal(loc=0.75, scale=0.1, size=len(weeks))
user_engagement_data = np.linspace(0.3, 0.9, len(weeks)) + np.random.normal(scale=0.05, size=len(weeks))
system_stability_data = np.linspace(0.9, 0.4, len(weeks)) + np.random.normal(scale=0.05, size=len(weeks))

# Extracted variables
xlabel = "Weeks"
ylabel_productivity = "Productivity"
ylabel_user_engagement = "User Engagement"
xlim = (0, 30)
ylim = (0, 1.2)
title = "Weekly Technology Metrics Analysis"
line_label_productivity = "Productivity"
line_label_user_engagement = "User Engagement"
line_label_system_stability = "System Stability"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Plot the productivity data on the primary y-axis
color = "#1f77b4"  # Blue
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_productivity, color=color)
(line1,) = ax1.plot(
    weeks,
    productivity_data,
    color=color,
    marker="o",
    linestyle="-",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_productivity,
)
ax1.tick_params(axis="y", labelcolor=color)
ax1.set_ylim(ylim)

# Create a secondary y-axis for the user engagement data
ax2 = ax1.twinx()
color = "#ff7f0e"  # Orange
ax2.set_ylabel(ylabel_user_engagement, color=color)
(line2,) = ax2.plot(
    weeks,
    user_engagement_data,
    color=color,
    marker="s",
    linestyle="--",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_user_engagement,
)
ax2.tick_params(axis="y", labelcolor=color)
ax2.set_ylim(ylim)

# Add a third line for system stability data
color = "#2ca02c"  # Green
(line3,) = ax1.plot(
    weeks,
    system_stability_data,
    color=color,
    marker="^",
    linestyle=":",
    linewidth=2,
    clip_on=False,
    zorder=10,
    label=line_label_system_stability,
)

# Customize the plot with a title, grid, and background color
fig.patch.set_facecolor("#f4f4f4")
ax1.set_facecolor("#e5f5f9")
ax2.set_facecolor("#f9e5e6")
ax1.set_xlim(xlim)
ax1.tick_params(axis="both", which="both", length=0)
ax2.tick_params(axis="both", which="both", length=0)
lines = [line1, line2, line3]
labels = [line.get_label() for line in lines]
fig.legend(
    lines, labels, loc="upper center", bbox_to_anchor=(0.5, 0.95), ncol=3, frameon=False
)
# Set the title and display the plot
plt.title(title, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_256.pdf", bbox_inches="tight")
