
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for the plot
sensors = np.array([2, 3, 4, 5, 6, 7, 8])
temperature = np.clip(
    np.linspace(18, 25, len(sensors)) + np.random.normal(0, 0.5, len(sensors)),
    18,
    None,
)
co2_concentration = np.clip(
    np.linspace(450, 400, len(sensors)) + np.random.normal(0, 5, len(sensors)),
    400,
    450,
)
normal_temp = [22] * len(sensors)  # Normal temperature baseline
safe_co2 = [420] * len(sensors)  # Safe CO2 level baseline

# Axes Limits and Labels
xlabel_value = "Number of Sensors"
ylabel_value_1 = "Average Temperature (°C)"
ylabel_value_2 = "CO2 Concentration (ppm)"
ylim_values_1 = [17, 27]
ylim_values_2 = [390, 460]
yticks_values_1 = range(17, 28, 1)
yticks_values_2 = range(390, 460, 10)

# Labels
label_1 = "Average Temperature"
label_2 = "CO2 Concentration"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(6, 5))

# Temperature and CO2 Concentration plots
(temp_line,) = ax1.plot(
    sensors,
    temperature,
    "o-",
    color="#2ca02c",  # Green
    label=label_1,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color="#2ca02c")
ax1.tick_params(
    axis="y", labelcolor="#2ca02c", direction="in", rotation=90, labelsize=12
)
ax1.tick_params(axis="x", direction="in", labelsize=12)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)

ax2 = ax1.twinx()
(co2_line,) = ax2.plot(
    sensors,
    co2_concentration,
    "^-",
    color="#1f77b4",  # Blue
    label=label_2,
    markersize=10,
    linewidth=3,
    mec="black",
    linestyle="--"
)
ax2.set_ylabel(ylabel_value_2, color="#1f77b4", fontsize=14)
ax2.tick_params(
    axis="y", labelcolor="#1f77b4", direction="in", rotation=90, labelsize=12
)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# Normal temperature and safe CO2 level baselines
ax1.axhline(y=22, color="#2ca02c", linestyle=":", linewidth=2)
ax1.axhline(y=420, color="#1f77b4", linestyle="--", linewidth=2)

# Custom legend for the plot
normal_temp_legend = Line2D([0], [0], color="black", linestyle=":", linewidth=2, label="Normal Temp")
safe_co2_legend = Line2D([0], [0], color="black", linestyle="--", linewidth=2, label="Safe CO2")

# Add the legend to the plot
first_legend = ax1.legend(
    handles=[normal_temp_legend, safe_co2_legend],
    loc="upper left",
    ncol=2,
    fontsize=14,
    edgecolor="black",
)
ax1.add_artist(first_legend)

# ===================
# Part 4: Saving Output
# ===================
# Add the second legend to the plot
plt.tight_layout()
plt.savefig("line_269.pdf", bbox_inches="tight")
