
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Data for the plot
months = np.array([1, 2, 3, 4, 5, 6, 7])
# Generate temperature data with a slight upward trend and some noise
temperature = np.clip(
    np.linspace(5, 25, len(months)) + np.random.normal(0, 2, len(months)), 
    0, 
    30
)
# Generate humidity data with a slight downward trend and some noise
humidity = np.clip(
    np.linspace(80, 60, len(months)) + np.random.normal(0, 5, len(months)), 
    50, 
    90
)
# Constant high and low temperatures to indicate extreme weather conditions
high_temp = [28, 29, 30, 30, 29, 28, 27]
low_temp = [0, 1, 2, 3, 4, 5, 6]

# Axes Limits and Labels
xlabel_value = "Months"

ylabel_value_1 = "Temperature (°C)"
ylabel_value_2 = "Humidity (%)"
ylim_values_1 = [0, 35]
ylim_values_2 = [50, 100]
yticks_values_1 = np.arange(0, 36, 5)
yticks_values_2 = np.arange(50, 101, 10)

# Labels
label_1 = "Temperature Trend"
label_2 = "Humidity Trend"
title="Weather Analysis: Temperature and Humidity Trends"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Temperature and Humidity plots
(temp_line,) = ax1.plot(
    months,
    temperature,
    color="#1f77b4",
    label=label_1,
    markersize=8,
    linewidth=2,
    mec="black",
    linestyle="-",
    marker="o",
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color="#1f77b4")
ax1.tick_params(
    axis="y", labelcolor="#1f77b4", direction="in", rotation=0, labelsize=12
)
ax1.tick_params(axis="x", direction="in", labelsize=12)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)

# Create a secondary y-axis for Humidity
ax2 = ax1.twinx()
(humidity_line,) = ax2.plot(
    months,
    humidity,
    color="#ff7f0e",
    label=label_2,
    markersize=8,
    linewidth=2,
    mec="black",
    linestyle="--",
    marker="s",
)
ax2.set_ylabel(ylabel_value_2, color="#ff7f0e", fontsize=14)
ax2.tick_params(
    axis="y", labelcolor="#ff7f0e", direction="in", rotation=0, labelsize=12
)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# High and Low temperature constant trends
ax1.plot(months, high_temp, ":", color="red", linewidth=2, label="High Temp")
ax1.plot(months, low_temp, "--", color="blue", linewidth=2, label="Low Temp")

# Custom legend for the plot
high_temp_legend = Line2D(
    [0], [0], color="red", linestyle=":", linewidth=2, label="High Temp"
)
low_temp_legend = Line2D(
    [0], [0], color="blue", linestyle="--", linewidth=2, label="Low Temp"
)

# Add the legend to the plot
first_legend = ax1.legend(
    handles=[high_temp_legend, low_temp_legend],
    loc="upper left",
    ncol=1,
    fontsize=12,
    edgecolor="black",
)
ax1.add_artist(first_legend)  # Add the first legend back to the plot
second_legend = ax1.legend(
    handles=[temp_line, humidity_line], loc="upper right", fontsize=12, edgecolor="black"
)  # Add the second legend

# Title and layout adjustments
plt.title(title, fontsize=16, pad=20)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
plt.savefig("line_274.pdf", bbox_inches="tight")
