
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.ticker import FuncFormatter, FixedLocator

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data for plotting (Years vs. Pollutant Levels)
years = np.array([1990, 1995, 2000, 2005, 2010, 2015, 2020])
co2_levels = np.array([350, 360, 370, 380, 390, 400, 410])
no2_levels = np.array([30, 28, 27, 26, 25, 24, 23])

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value = "Pollutant Levels (ppm)"

yticks_values = [20, 40, 100, 200, 300, 400, 500]
yticks_labels = [
    "20",
    "40",
    "100",
    "200",
    "300",
    "400",
    "500",
]
ylim_values = [15, 450]

axvline_x = 2000

# Labels
label_CO2 = "CO2 Levels"
label_NO2 = "NO2 Levels"
title = "Pollutant Levels Over Time"
legend_CO2 = "CO2"
legend_NO2 = "NO2"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting the data
plt.figure(figsize=(9, 6))
plt.plot(
    years,
    co2_levels,
    "o-",
    label=label_CO2,
    color="#2ca02c",  # Green color for CO2
    linewidth=1.4,
    markersize=4,
)
plt.plot(
    years,
    no2_levels,
    "s-",
    label=label_NO2,
    color="#8c564b",  # Brown color for NO2
    linewidth=1.4,
    markersize=4,
)

# Setting the x-axis and y-axis to log scale
plt.xscale("log")
plt.yscale("log")

# Set y-axis to only display specific ticks and extend y-axis to leave space at top
plt.yticks(
    yticks_values,
    yticks_labels,
)
plt.ylim(ylim_values)  # Extend y-axis to leave some space above the highest value

# Disable the automatic grid for x-axis
plt.grid(True, which="both", ls="--", axis="y")  # Only enable y-axis grid

# Manually add a grid line for x=2000
plt.axvline(x=axvline_x, color="grey", linestyle="--", linewidth=0.5)

# Adjusting x-axis ticks to show significant years
plt.gca().xaxis.set_major_locator(FixedLocator(years))

# Formatting the x-axis and y-axis tick labels
plt.gca().xaxis.set_major_formatter(FuncFormatter(lambda value, _: "{:.0f}".format(value)))
plt.gca().yaxis.set_major_formatter(FuncFormatter(lambda y, _: "{:.0e}".format(y)))

# Adding labels and title
plt.xlabel(xlabel_value, fontsize=12)
plt.ylabel(ylabel_value, fontsize=12)
plt.title(title, fontsize=14)

# Adding a legend at the center right
plt.legend(loc="center left", bbox_to_anchor=(0.67, 0.5), fontsize=18)

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout to reduce white space
plt.tight_layout()
plt.savefig("line_95.pdf", bbox_inches="tight")
