
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.ticker import FuncFormatter, FixedLocator

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data for plotting
years = np.array([2000, 2005, 2010, 2015, 2020])
fuel_consumption = np.array([8, 7, 6, 5, 4])  # liters per 100 km
ev_range = np.array([100, 150, 200, 300, 400])  # km

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value = "Value"

yticks_values = [1, 10, 100, 1000]
yticks_labels = ["1", "10", "100", "1000"]
ylim_values = [1, 1000]

axvline_x = 2010

# Labels
label_FuelConsumption = "Fuel Consumption (l/100 km)"
label_EVRange = "Electric Vehicle Range (km)"
chart_title = "Transportation Trends Over Years"
legend_title = "Metric"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting the data
plt.figure(figsize=(10, 7))
plt.plot(
    years,
    fuel_consumption,
    "o-",
    label=label_FuelConsumption,
    color="#2a9d8f",  # Nice shade of green
    linewidth=2,
    markersize=6,
)
plt.plot(
    years,
    ev_range,
    "s--",
    label=label_EVRange,
    color="#e76f51",  # Nice shade of red-orange
    linewidth=2,
    markersize=6,
)

# Setting the y-axis to log scale
plt.yscale("log")

# Set y-axis to only display specific ticks and extend y-axis to leave space at top
plt.yticks(
    yticks_values,
    yticks_labels,
)
plt.ylim(ylim_values)  # Extend y-axis to leave some space above 10^-1

# Enable grid for y-axis
plt.grid(True, which="both", ls="--", axis="y")

# Manually add a vertical line for the year 2010
plt.axvline(x=axvline_x, color="grey", linestyle="--", linewidth=1)

# Adjusting x-axis ticks to only show the years of data points
plt.gca().xaxis.set_major_locator(FixedLocator(years))

# Formatting the y-axis tick labels
plt.gca().yaxis.set_major_formatter(FuncFormatter(lambda y, _: "{:.0f}".format(y)))

# Adding labels and title
plt.xlabel(xlabel_value, fontsize=14)
plt.ylabel(ylabel_value, fontsize=14)
plt.title(chart_title, fontsize=16)

# Adding a legend at the lower right
plt.legend(title=legend_title, loc="lower right", fontsize=12, title_fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout to reduce white space
plt.tight_layout()
plt.savefig("line_97.pdf", bbox_inches="tight")
