
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Sample data to represent tech gadgets
gadgets = ["Smartphone", "Laptop", "Smartwatch", "Tablet", "Gaming Console"]
prices = [700, 1200, 350, 500, 450]  # Average price per item for each gadget
popularity = [9.0, 8.5, 7.0, 7.5, 8.0]  # Popularity index out of 10

# Data for violin plot; customer satisfaction scores (1-10 scale)
satisfaction_data = np.random.normal(loc=[8.5, 7.0, 6.5, 7.0, 8.0], scale=0.5, size=(50, 5))

# Labels and titles
ax0xlabel = "Average Price ($)"
ax0ylabel = "Popularity Index"
ax0title = "Tech Gadget Popularity vs Price"
ax1xticks = range(len(gadgets))
ax1xlabel = "Gadgets"
ax1ylabel = "Customer Satisfaction"
ax1title = "Customer Satisfaction Distribution Across Gadgets"

x = np.linspace(1, 10, 300)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 1x2 subplot layout
fig = plt.figure(figsize=(12, 7))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 2])

# Scatter plot on the left
ax0 = fig.add_subplot(gs[0])
sc = ax0.scatter(
    prices, popularity, s=100, marker='o', c=np.linspace(0.1, 0.9, len(gadgets)), cmap="winter"
)
for i, gadget in enumerate(gadgets):
    ax0.text(prices[i], popularity[i] + 0.05, gadget, fontsize=9, style='italic')
ax0.set_xlabel(ax0xlabel)
ax0.set_ylabel(ax0ylabel)
ax0.set_title(ax0title)

# Violin plot on the right
ax1 = fig.add_subplot(gs[1])

# Creating half-violins
for i, gadget in enumerate(gadgets):
    kde = gaussian_kde(satisfaction_data[:, i])
    y = kde(x)
    max_y = max(y)
    ax1.fill_betweenx(x, -y / max_y * 0.5 + i, i, color="lightgreen", alpha=0.5)
    ax1.fill_betweenx(x, y / max_y * 0.5 + i, i, color="green", alpha=0.5)

ax1.set_xticks(ax1xticks)
ax1.set_xticklabels(gadgets)
ax1.set_xlabel(ax1xlabel)
ax1.set_ylabel(ax1ylabel)
ax1.set_title(ax1title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
# Show plot
plt.savefig("multidiff_41.pdf", bbox_inches="tight")
