
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data to represent media streaming services
companies = ["Netflix", "Hulu", "Disney+", "HBO Max", "Amazon Prime"]
prices = [15.99, 11.99, 7.99, 14.99, 8.99]  # Average monthly price for each service
ratings = [8.4, 7.8, 8.1, 7.5, 7.9]  # User rating out of 10

# Data for violin plot; user satisfaction scores (1-10 scale), different distribution means
satisfaction_data = np.random.normal(loc=[8.5, 7.0, 8.0, 7.5, 7.8], scale=0.5, size=(50, 5))

ax0xlabel = "Average Monthly Price ($)"
ax0ylabel = "User Rating (out of 10)"
ax0title = "Streaming Service User Ratings vs Price"
ax1xticks = range(len(companies))
ax1xlabel = "Streaming Services"
ax1ylabel = "User Satisfaction"
ax1title = "User Satisfaction Distribution Across Streaming Services"
x = np.linspace(1, 10, 300)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 1x2 subplot layout
fig = plt.figure(figsize=(12, 6))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 2])

# Scatter plot on the left
ax0 = fig.add_subplot(gs[0])
sc = ax0.scatter(
    prices, ratings, s=100, c=np.linspace(0.1, 0.9, len(companies)), cmap="plasma"
)
for i, company in enumerate(companies):
    ax0.text(prices[i]-0.03, ratings[i] - 0.03, company, fontsize=9)
ax0.set_xlabel(ax0xlabel)
ax0.set_ylabel(ax0ylabel)
ax0.set_title(ax0title)

# Violin plot on the right
ax1 = fig.add_subplot(gs[1])

# Creating half-violins with different colors for each company
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd"]
for i, company in enumerate(companies):
    kde = gaussian_kde(satisfaction_data[:, i])
    y = kde(x)
    max_y = max(y)
    ax1.fill_betweenx(x, -y / max_y * 0.5 + i, i, color=colors[i], alpha=0.5)
    ax1.fill_betweenx(x, y / max_y * 0.5 + i, i, color=colors[i], alpha=0.7)

ax1.set_xticks(ax1xticks)
ax1.set_xticklabels(companies)
ax1.set_xlabel(ax1xlabel)
ax1.set_ylabel(ax1ylabel)
ax1.set_title(ax1title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
# Show plot
plt.savefig("multidiff_43.pdf", bbox_inches="tight")
