
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data for the first plot (bar plot with white circles)
energy_types = [
    "Solar",
    "Wind",
    "Hydro",
    "Nuclear",
    "Coal",
    "Oil",
    "Gas",
    "Geothermal",
    "Biomass",
    "Wave",
    "Tidal",
    "Solar",
    "Wind",
    "Hydro",
    "Nuclear",
    "Coal",
]
n_categories = len(energy_types)

energy_efficiency_correct = np.random.uniform(50, 100, n_categories)
energy_efficiency_incorrect = energy_efficiency_correct - np.random.uniform(5, 20, n_categories)
sorted_indices = np.argsort((energy_efficiency_correct + energy_efficiency_incorrect) / 2)
categories = np.array(energy_types)[sorted_indices]
energy_efficiency_correct = energy_efficiency_correct[sorted_indices]
energy_efficiency_incorrect = energy_efficiency_incorrect[sorted_indices]

assignments = ["Region A", "Region B", "Region C", "Region D"]

# Data for the second plot (error bar plot with legend)
for i in range(len(assignments)):
    aggregate_efficiency = np.random.randint(-250, 250, size=5)
    correct_rating = np.random.randint(0, 300, size=5)
    error = np.random.randint(10, 15, size=5)

# Data for the third plot (scatter plot with trend line and correct text)
aggregate_efficiencies = [
    np.linspace(180, 200, 5) - 150 * i + np.random.randint(-20, 20, 5)
    for i in range(len(assignments))
]
judge_accuracies = [
    np.linspace(80, 85, 5) - 10 * i + np.random.randint(-20, 20, 5)
    for i in range(len(assignments))
]
scatterlabels = ["Actual Efficiency", "Predicted Efficiency"]
xlabels = ["Efficiency Rating", "Aggregate Efficiency Rating", "Aggregate Efficiency Rating"]
ylabels = ["Energy Type", "Correct Rating - Incorrect Rating", "Judgment Accuracy (%)"]

ax2textlabel = "Region: Region D"
title = "Energy Efficiency Analysis"
text_j = 200
text_i = 38

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Adjusting figure size to match the original image's dimensions
fig, axs = plt.subplots(1, 3, figsize=(10, 4))

# Colors and assignments for the first plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728"]

# First plot (bar plot with white circles)
axs[0].grid(axis="x", zorder=0)
# Plotting the lines first
for i in range(n_categories):
    color = colors[i // 4]
    axs[0].plot(
        [energy_efficiency_incorrect[i], energy_efficiency_correct[i]], [i, i], color=color, lw=6, zorder=1
    )  # Colored lines for each category
# Plotting the points
axs[0].scatter(
    energy_efficiency_correct,
    range(n_categories),
    color="black",
    label=scatterlabels[0],
    zorder=2,
    s=80,
)  # Black dots for 'Actual Efficiency'
axs[0].scatter(
    energy_efficiency_incorrect,
    range(n_categories),
    color="white",
    edgecolors="black",
    label=scatterlabels[1],
    zorder=2,
    s=80,
)  # White dots for 'Predicted Efficiency'
axs[0].legend(loc="lower right", title="Measurement")
axs[0].set_xlabel(xlabels[0])
axs[0].set_ylabel(ylabels[0])
axs[0].set_yticks(range(n_categories))
axs[0].set_yticklabels(categories[::-1])

# Second plot (error bar plot with legend)
for i in range(len(assignments)):
    axs[1].errorbar(
        aggregate_efficiency,
        correct_rating,
        yerr=error,
        xerr=error + 10,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
axs[1].set_xlabel(xlabels[1])
axs[1].set_ylabel(ylabels[1])
axs[1].grid()

# Third plot (scatter plot with trend line and correct text)
for i in range(len(assignments)):
    aggregate_efficiency = aggregate_efficiencies[i]
    judge_accuracy = judge_accuracies[i]
    error = np.random.randint(5, 10, size=5)
    axs[2].errorbar(
        aggregate_efficiency,
        judge_accuracy,
        yerr=error,
        xerr=error + 10,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
# Plotting the trend line
axs[2].plot(
    np.linspace(-300, 200, 5), np.linspace(40, 85, 5), color="black", lw=2, ls="-"
)
axs[2].set_xlabel(xlabels[2])
axs[2].set_ylabel(ylabels[2])
# set box edge color
axs[2].text(
    text_j,
    text_i,
    ax2textlabel,
    fontsize=8,
    color="black",
    ha="right",
    va="bottom",
    bbox=dict(facecolor="white", edgecolor="black"),
)
axs[2].grid()

# add legend to the whole figure and set colormaps
legend_handles = [
    mlines.Line2D(
        [], [], color=color, marker="s", linestyle="None", markersize=10, label=label
    )
    for label, color in zip(assignments, colors)
]
fig.legend(
    handles=legend_handles,
    loc="upper center",
    title=title,
    ncol=len(assignments),
    bbox_to_anchor=(0.5, 1.15),
    facecolor="white",
)

# ===================
# Part 4: Saving Output
# ===================
# Show plot
plt.tight_layout()
plt.savefig("multidiff_51.pdf", bbox_inches="tight")
