
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(0)

# Generating synthetic data for the violin plot
# Average number of messages sent per day in two age groups
age_group_teen = np.random.normal(50, 10, 150)  # Messages by teens
age_group_adult = np.random.normal(30, 8, 150)  # Messages by adults

# Generating synthetic data for the density plot
# Response times in milliseconds for three communication apps
app1_response_time = np.random.normal(200, 50, 1000)  # App 1 response time
app2_response_time = np.random.normal(250, 60, 1000)  # App 2 response time
app3_response_time = np.random.normal(300, 70, 1000)  # App 3 response time

xs = np.linspace(0, 600, 200)

labels = ["App 1", "App 2", "App 3"]
titles = ["Messages Sent Per Day", "App Response Time Distribution"]
ylabels = ["Number of Messages", "Density"]
ax1xticks = [1, 2]
ax1xtickslabels = ["Teens", "Adults"]
ax2xlabel = "Response Time (ms)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating the figure and axes
fig, axes = plt.subplots(nrows=1, ncols=2, figsize=(10, 5))

# Violin plot
violin_data = [age_group_teen, age_group_adult]
axes[0].violinplot(violin_data, showmeans=False, showmedians=True)
axes[0].set_title(titles[0])
axes[0].set_ylabel(ylabels[0])
axes[0].set_xticks(ax1xticks)
axes[0].set_xticklabels(ax1xtickslabels)
axes[0].grid(True)

# Density plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c"]  # Custom colors

for data, color, label in zip([app1_response_time, app2_response_time, app3_response_time], colors, labels):
    density = gaussian_kde(data)
    axes[1].fill_between(xs, density(xs), color=color, alpha=0.5, label=label)
axes[1].set_title(titles[1])
axes[1].set_xlabel(ax2xlabel)
axes[1].set_ylabel(ylabels[1])
axes[1].legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout for better visual appearance
plt.tight_layout()
plt.savefig("multidiff_83.pdf", bbox_inches="tight")
