
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(42)

# Generating synthetic data for the violin plot
# Average annual screen time (hours) for two age groups over different years
age_group1_screen_time = np.random.normal(3, 0.5, 150)  # Age group 1
age_group2_screen_time = np.random.normal(5, 0.8, 150)  # Age group 2

# Generating synthetic data for the density plot
# Distribution of daily active users (in millions) for three app types
social_apps = np.random.normal(50, 10, 1000)  # Social media apps
game_apps = np.random.normal(30, 8, 1000)  # Gaming apps
utility_apps = np.random.normal(10, 5, 1000)  # Utility apps

xs = np.linspace(0, 100, 200)

labels = ["Social Apps", "Gaming Apps", "Utility Apps"]
titles = ["Average Annual Screen Time", "Daily Active Users Distribution"]
ylabels = ["Time (hours)", "Density"]
ax1xticks = [1, 2]
ax1xtickslabels = ["Age Group 1", "Age Group 2"]
ax2xlabel = "Daily Active Users (millions)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating the figure and axes
fig, axes = plt.subplots(nrows=1, ncols=2, figsize=(12, 6))

# Violin plot
violin_data = [age_group1_screen_time, age_group2_screen_time]
axes[0].violinplot(violin_data, showmeans=False, showmedians=True)
axes[0].set_title(titles[0])
axes[0].set_ylabel(ylabels[0])
axes[0].set_xticks(ax1xticks)
axes[0].set_xticklabels(ax1xtickslabels)
axes[0].grid(True, linestyle='--', linewidth=0.5, color='gray')

# Density plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c"]  # Choosing a modern tech-oriented color scheme

for data, color, label in zip([social_apps, game_apps, utility_apps], colors, labels):
    density = gaussian_kde(data)
    axes[1].fill_between(xs, density(xs), color=color, alpha=0.4, label=label)
    axes[1].plot(xs, density(xs), color=color, linestyle='-', linewidth=1.5)
axes[1].set_title(titles[1])
axes[1].set_xlabel(ax2xlabel)
axes[1].set_ylabel(ylabels[1])
axes[1].legend()
axes[1].grid(True, linestyle='--', linewidth=0.5, color='gray')

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout for better visual appearance
plt.tight_layout()
plt.savefig("multidiff_84.pdf", bbox_inches="tight")
