
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(0)
# Generate synthetic legal data
# Case counts at different times of the year
q1_cases = np.random.poisson(150, 500)  # Q1 case counts
q4_cases = np.random.poisson(200, 500)  # Q4 case counts

# Verdict time data for different case types
criminal_verdicts = np.random.normal(30, 5, 1000)  # Criminal case verdict times
civil_verdicts = np.random.normal(60, 10, 1000)  # Civil case verdict times

# Impact of law change over time
time_period = np.linspace(0, 5, 1000)  # Time in years
pending_cases = np.exp(-0.5 * time_period) * 1000  # Exponential decay

# Labels and titles for the plots
ax1labels = ["Q1 Cases", "Q4 Cases"]
titles = [
    "Number of Cases by Quarter",
    "Verdict Time Distribution by Case Type",
    "Impact of Law Change on Pending Cases",
]
xlabels = ["Number of Cases", "Time (Years)"]
ylabels = ["Frequency", "Verdict Time (Days)", "Pending Cases"]
ax2xtickslabels = ["Criminal", "Civil"]
ax2xticks = [1, 2]
bins = np.linspace(50, 250, 21)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure using GridSpec
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Histogram plot across top (1,1 and 1,2)
ax1 = fig.add_subplot(gs[0, :])
# Define bin edges and width to align bars side by side

ax1.hist(
    q1_cases,
    bins=bins,
    alpha=0.7,
    label=ax1labels[0],
    color="#3b75af",
    edgecolor="black",
)
ax1.hist(
    q4_cases,
    bins=bins,
    alpha=0.7,
    label=ax1labels[1],
    color="#88c0d0",
    edgecolor="black",
)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.legend()

# Violin plot on bottom left (2,1)
ax2 = fig.add_subplot(gs[1, 0])
ax2.violinplot([criminal_verdicts, civil_verdicts], showmeans=False, showmedians=True)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_xticks(ax2xticks)
ax2.set_xticklabels(ax2xtickslabels)
ax2.grid(True)

# Fill between plot on bottom right (2,2)
ax3 = fig.add_subplot(gs[1, 1])
ax3.fill_between(time_period, pending_cases, color="green", alpha=0.2)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[1])
ax3.set_ylabel(ylabels[2])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for a better visual appearance
plt.tight_layout()
plt.savefig("multidiff_93.pdf", bbox_inches="tight")
